package STIX::Malware;

use 5.010001;
use strict;
use warnings;
use utf8;

use STIX::Common::List;
use STIX::Common::OpenVocabulary;
use Types::Standard qw(Str Num Enum Bool InstanceOf);
use Types::TypeTiny qw(ArrayLike);

use Moo;
use namespace::autoclean;

extends 'STIX::Common::Properties';

use constant SCHEMA =>
    'http://raw.githubusercontent.com/oasis-open/cti-stix2-json-schemas/stix2.1/schemas/sdos/malware.json';

use constant PROPERTIES => (
    qw(type spec_version id created modified),
    qw(created_by_ref revoked labels confidence lang external_references object_marking_refs granular_markings extensions),
    qw(name description malware_types is_family aliases kill_chain_phases first_seen last_seen operating_system_refs architecture_execution_envs implementation_languages capabilities sample_refs)
);

use constant STIX_OBJECT      => 'SDO';
use constant STIX_OBJECT_TYPE => 'malware';

has name        => (is => 'rw', isa => Str);
has description => (is => 'rw', isa => Str);

has malware_types => (
    is      => 'rw',
    isa     => ArrayLike [Enum [STIX::Common::OpenVocabulary->MALWARE_TYPE()]],
    default => sub { STIX::Common::List->new }
);

has is_family => (is => 'rw', isa => Bool, required => 1);
has aliases => (is => 'rw', isa => ArrayLike [Str], default => sub { STIX::Common::List->new });
has kill_chain_phases => (
    is      => 'rw',
    isa     => ArrayLike [InstanceOf ['STIX::Common::KillChainPhase']],
    default => sub { STIX::Common::List->new }
);

has first_seen => (
    is     => 'rw',
    isa    => InstanceOf ['STIX::Common::Timestamp'],
    coerce => sub { ref($_[0]) ? $_[0] : STIX::Common::Timestamp->new($_[0]) },
);

has last_seen => (
    is     => 'rw',
    isa    => InstanceOf ['STIX::Common::Timestamp'],
    coerce => sub { ref($_[0]) ? $_[0] : STIX::Common::Timestamp->new($_[0]) },
);

has operating_system_refs => (is => 'rw', isa => ArrayLike [Str], default => sub { STIX::Common::List->new });

has architecture_execution_envs => (
    is      => 'rw',
    isa     => ArrayLike [Enum [STIX::Common::OpenVocabulary->PROCESSOR_ARCHITECTURE()]],
    default => sub { STIX::Common::List->new }
);

has implementation_languages => (
    is      => 'rw',
    isa     => ArrayLike [Enum [STIX::Common::OpenVocabulary->IMPLEMENTATION_LANGUAGE()]],
    default => sub { STIX::Common::List->new }
);

has capabilities => (
    is      => 'rw',
    isa     => ArrayLike [Enum [STIX::Common::OpenVocabulary->MALWARE_CAPABILITIES()]],
    default => sub { STIX::Common::List->new }
);

has sample_refs => (
    is      => 'rw',
    isa     => ArrayLike [InstanceOf ['STIX::Observable', 'STIX::Common::Identifier']],
    default => sub { STIX::Common::List->new }
);


1;

=encoding utf-8

=head1 NAME

STIX::Malware - STIX Domain Object (SDO) - Malware

=head1 SYNOPSIS

    use STIX::Malware;

    my $malware = STIX::Malware->new();


=head1 DESCRIPTION

Malware is a type of TTP that is also known as malicious code and malicious
software, refers to a program that is inserted into a system, usually
covertly, with the intent of compromising the confidentiality, integrity,
or availability of the victim's data, applications, or operating system
(OS) or of otherwise annoying or disrupting the victim.


=head2 METHODS

L<STIX::Malware> inherits all methods from L<STIX::Common::Properties>
and implements the following new ones.

=over

=item STIX::Malware->new(%properties)

Create a new instance of L<STIX::Malware>.

=item $malware->aliases

Alternative names used to identify this Malware or Malware family.

=item $malware->architecture_execution_envs

The processor architectures (e.g., x86, ARM, etc.) that the malware
instance or family is executable on.
(See C<PROCESSOR_ARCHITECTURE> in L<STIX::Common::OpenVocabulary>)

=item $malware->capabilities

Specifies any capabilities identified for the malware instance or family.
(See C<MALWARE_CAPABILITIES> in L<STIX::Common::OpenVocabulary>).

=item $malware->description

Provides more context and details about the Malware object.

=item $malware->first_seen

The time that the malware instance or family was first seen.

=item $malware->id

=item $malware->implementation_languages

The programming language(s) used to implement the malware instance or
family. (See C<IMPLEMENTATION_LANGUAGE> in L<STIX::Common::OpenVocabulary>).

=item $malware->kill_chain_phases

The list of kill chain phases for which this Malware instance can be used.

=item $malware->last_seen

The time that the malware family or malware instance was last seen.

=item $malware->malware_types

The type of malware being described.
(See C<MALWARE_TYPE> in L<STIX::Common::OpenVocabulary>)

=item $malware->name

The name used to identify the Malware.

=item $malware->operating_system_refs

The operating systems that the malware family or malware instance is
executable on.

=item $malware->sample_refs

The sample_refs property specifies a list of identifiers of the SCO file or
artifact objects associated with this malware instance(s) or family.

=item $malware->type

The type of this object, which MUST be the literal C<malware>.

=back


=head2 HELPERS

=over

=item $malware->TO_JSON

Encode the object in JSON.

=item $malware->to_hash

Return the object HASH.

=item $malware->to_string

Encode the object in JSON.

=item $malware->validate

Validate the object using JSON Schema (see L<STIX::Schema>).

=back


=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the issue tracker
at L<https://github.com/giterlizzi/perl-STIX/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/giterlizzi/perl-STIX>

    git clone https://github.com/giterlizzi/perl-STIX.git


=head1 AUTHOR

=over 4

=item * Giuseppe Di Terlizzi <gdt@cpan.org>

=back


=head1 LICENSE AND COPYRIGHT

This software is copyright (c) 2024 by Giuseppe Di Terlizzi.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
