import textwrap
from typing import List, TYPE_CHECKING
from collections.abc import Sequence

import pytest

from debputy.lsp.languages.lsp_debian_control import _reformat_debian_control
from debputy.lsp.maint_prefs import MaintainerPreferenceTable
from debputy.lsp.text_edit import apply_text_edits
from debputy.packages import DctrlParser
from debputy.plugin.api.feature_set import PluginProvidedFeatureSet
from lint_tests.lint_tutil import (
    ReformatWrapper,
    apply_formatting_edits,
)

if TYPE_CHECKING:
    import lsprotocol.types as types
else:
    import debputy.lsprotocol.types as types


ALL_REQUIRED_FILES = [
    "debian/changelog",
    "debian/copyright",
]


@pytest.fixture
def reformater(
    debputy_plugin_feature_set: PluginProvidedFeatureSet,
    lint_dctrl_parser: DctrlParser,
    maintainer_preference_table: MaintainerPreferenceTable,
) -> ReformatWrapper:

    return ReformatWrapper(
        "/nowhere/debian/control",
        _reformat_debian_control,
        debputy_plugin_feature_set,
        lint_dctrl_parser,
        maintainer_preference_table,
    )


def test_dctrl_reformat(reformater: ReformatWrapper) -> None:
    lines = textwrap.dedent(
        """\
    Source: foo
    Some-Other-Field: bar
    Build-Depends: debhelper-compat (= 13)
    Homepage: https://example.com

    Package: foo
    Architecture: all
    Section: devel
    """
    ).splitlines(keepends=True)

    edits = reformater.reformat(lines)
    # By default, we do nothing
    assert not edits

    edits_black = reformater.reformat_with_named_style("black", lines)
    assert edits_black

    actual_reformatted_black = apply_formatting_edits(lines, edits_black)

    expected_reformatted_black = textwrap.dedent(
        """\
    Source: foo
    Some-Other-Field: bar
    Build-Depends:
     debhelper-compat (= 13),
    Homepage: https://example.com

    Package: foo
    Architecture: all
    Section: devel
    """
    )

    assert actual_reformatted_black == expected_reformatted_black


def test_dctrl_reformat_canonicalize_field_names(reformater: ReformatWrapper) -> None:
    lines = textwrap.dedent(
        """\
    source: foo
    # This field will not change case, since it is unknown
    some-other-field: bar
    BUILD-DEPENDS:
     debhelper-compat (= 13),
    # In memory of Steve (#1068220)
    HoMEpaGe: https://example.com

    package: foo
    architecture: all
    Section: devel
    """
    ).splitlines(keepends=True)

    edits = reformater.reformat(lines)
    # By default, we do nothing
    assert not edits

    edits_black = reformater.reformat_with_named_style("black", lines)
    assert edits_black

    actual_reformatted_black = apply_formatting_edits(lines, edits_black)

    expected_reformatted_black = textwrap.dedent(
        """\
    Source: foo
    # This field will not change case, since it is unknown
    some-other-field: bar
    Build-Depends:
     debhelper-compat (= 13),
    # In memory of Steve (#1068220)
    Homepage: https://example.com

    Package: foo
    Architecture: all
    Section: devel
    """
    )

    assert actual_reformatted_black == expected_reformatted_black


def test_dctrl_reformat_preserve_comments(reformater: ReformatWrapper) -> None:
    # Based on an example observed in  sbuild`. Thanks to zeha fore reporting.
    lines = textwrap.dedent(
        """\
    Source: foo
    # Test
    Build-Depends:
     foo,
     bar,

    Package: foo
    # Test
    architecture: i386 amd64
    Section: devel
    # Test
    Depends: foo
    # Test
    recommends:
     bar,
    """
    ).splitlines(keepends=True)

    edits = reformater.reformat(lines)
    # By default, we do nothing
    assert not edits

    edits_black = reformater.reformat_with_named_style("black", lines)
    assert edits_black

    actual_reformatted_black = apply_formatting_edits(lines, edits_black)

    expected_reformatted_black = textwrap.dedent(
        """\
    Source: foo
    # Test
    Build-Depends:
     bar,
     foo,

    Package: foo
    # Test
    Architecture: amd64 i386
    Section: devel
    # Test
    Depends:
     foo,
    # Test
    Recommends:
     bar,
    """
    )

    assert actual_reformatted_black == expected_reformatted_black
