/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
/* Largely inspired of xmms control socket stuff */

/* required for getsubopt(); the __sun test gives us strncasecmp() on solaris */
#if ! defined (__sun) && ! defined (__OpenBSD__) && ! defined (__FreeBSD__) && !defined(__NetBSD__) && !defined(__DragonFly__)
#define _XOPEN_SOURCE 500
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>

#include "common.h"
#include "xine-toolkit/backend.h"
#include "mediamark.h"
#include "session.h"
#include "panel.h"
#include "playlist.h"
#include "actions.h"
#include "event.h"
#include "session_internal.h"

struct xui_session_s {
  gGui_t *gui;
  int session_id, ctrl_fd, going;
  pthread_t thread_server;
  char *socket_name;
  pthread_mutex_t mutex;
};

static int send_uint32(int session, ctrl_commands_t command, uint32_t value) {
  int fd;

  if((fd = connect_to_session(session)) == -1)
    return -1;
  if (send_packet(fd, command, &value, sizeof(uint32_t)) >= 0) {
    read_ack(fd);
  }
  close(fd);

  return 0;
}

static uint32_t get_uint32(int session, ctrl_commands_t command) {
  ctrl_header_packet_t  hdr;
  void                 *data;
  int                   fd, ret = 0;

  if((fd = connect_to_session(session)) == -1)
    return ret;

  if (send_packet(fd, command, NULL, 0) < 0) {
    close(fd);
    return 0;
  }
  data = read_packet(fd, &hdr);
  if(data) {
    ret = *((uint32_t *) data);
    free(data);
  }
  read_ack(fd);
  close(fd);

  return ret;
}

#if 0
void send_boolean(int session, ctrl_commands_t command, uint8_t value) {
  int     fd;
  uint8_t bvalue = (value > 0) ? 1 : 0;

  if((fd = connect_to_session(session)) == -1)
    return;
  send_packet(fd, command, &bvalue, sizeof(uint8_t));
  read_ack(fd);
  close(fd);
}

uint8_t get_boolean(int session, ctrl_commands_t command) {
  ctrl_header_packet_t  hdr;
  uint8_t               ret = 0;
  void                 *data;
  int                   fd;

  if((fd = connect_to_session(session)) == -1)
    return ret;

  send_packet(fd, command, NULL, 0);
  data = read_packet(fd, &hdr);
  if(data) {
    ret = *((uint8_t *) data);
    free(data);
  }
  read_ack(fd);
  close(fd);

  return ret;
}
#endif

static void *ctrlsocket_func (void *data) {
  xui_session_t *s = data;
  gGui_t *gui = s->gui;

  pthread_mutex_lock (&s->mutex);
  while (!s->going) {
    pthread_mutex_unlock (&s->mutex);
    xine_usec_sleep (10000);
    pthread_mutex_lock (&s->mutex);
  }

  while (s->going) {
    union {
      struct sockaddr_un un;
      struct sockaddr sa;
    } saddr;
    struct timeval tv = { .tv_sec = 0, .tv_usec = 100000 };
    fd_set set;
    serv_header_packet_t shdr;
    int fd;
    socklen_t len;
    int res, ack = 1;

    pthread_mutex_unlock (&s->mutex);

    FD_ZERO(&set);
    FD_SET (s->ctrl_fd, &set);

    res = select (s->ctrl_fd + 1, &set, NULL, NULL, &tv);
    if (res == 0) {
      pthread_mutex_lock (&s->mutex);
      continue;
    }
    if (res < 0) {
      if (errno == EINTR || errno == EAGAIN) {
        pthread_mutex_lock (&s->mutex);
        continue;
      }
      if (gui->verbosity >= 1) {
        int e = errno;
        printf ("gui.session.select.failed (%d, %s).\n", e, strerror (e));
      }
      pthread_mutex_lock (&s->mutex);
      break;
    }

    len = sizeof (saddr.un);
    fd = accept (s->ctrl_fd, &saddr.sa, &len);
    if (fd < 0) {
      pthread_mutex_lock (&s->mutex);
      continue;
    }

    if (fcntl(fd, F_SETFD, FD_CLOEXEC) < 0) {
      if (gui->verbosity >= 1) {
        int e = errno;
        printf ("gui.session.cloexec.failed (%d, %s).\n", e, strerror (e));
      }
    }

    if (read (fd, &(shdr.hdr), sizeof (ctrl_header_packet_t)) != (ssize_t)sizeof (ctrl_header_packet_t)) {
      close(fd);
      pthread_mutex_lock (&s->mutex);
      continue;
    }

    if (shdr.hdr.data_length) {
      shdr.data = malloc (shdr.hdr.data_length);
      if (read(fd, shdr.data, shdr.hdr.data_length) != (ssize_t)shdr.hdr.data_length) {
        SAFE_FREE( shdr.data);
        close(fd);
        pthread_mutex_lock (&s->mutex);
        continue;
      }
    }

    shdr.fd = fd;

    switch (shdr.hdr.command) {

      case CMD_PLAY:
        xitk_lock (gui->xitk, 1);
        gui_play (NULL, gui);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_SLOW_2:
        xine_set_param (gui->stream, XINE_PARAM_SPEED, XINE_SPEED_SLOW_2);
        break;

      case CMD_SLOW_4:
        xine_set_param (gui->stream, XINE_PARAM_SPEED, XINE_SPEED_SLOW_4);
        break;

      case CMD_PAUSE:
        xine_set_param (gui->stream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
        break;

      case CMD_FAST_2:
        xine_set_param (gui->stream, XINE_PARAM_SPEED, XINE_SPEED_FAST_2);
        break;

      case CMD_FAST_4:
        xine_set_param (gui->stream, XINE_PARAM_SPEED, XINE_SPEED_FAST_4);
        break;

      case CMD_STOP:
        xitk_lock (gui->xitk, 1);
        gui_stop (NULL, gui);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_QUIT:
        xitk_lock (gui->xitk, 1);
        gui_exit (NULL, gui);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_FULLSCREEN:
        xitk_lock (gui->xitk, 1);
        gui_set_fullscreen_mode (NULL, gui);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_EJECT:
        xitk_lock (gui->xitk, 1);
        gui_eject (NULL, gui);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_AUDIO_NEXT:
        xitk_lock (gui->xitk, 1);
        gui_nextprev_audio_channel (NULL, GUI_NEXT (gui));
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_AUDIO_PREV:
        xitk_lock (gui->xitk, 1);
        gui_nextprev_audio_channel (NULL, GUI_PREV (gui));
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_SPU_NEXT:
        xitk_lock (gui->xitk, 1);
        gui_nextprev_spu_channel (NULL, GUI_NEXT (gui));
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_SPU_PREV:
        xitk_lock (gui->xitk, 1);
        gui_nextprev_spu_channel (NULL, GUI_PREV (gui));
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_FIRST:
        xitk_lock (gui->xitk, 1);
        if (gui->playlist.num) {
          if (xine_get_status (gui->stream) == XINE_STATUS_PLAY)
            gui_stop (NULL, gui);
          gui->playlist.cur = 0;
          gui_current_set_index (gui, GUI_MMK_CURRENT);
          gui_play (NULL, gui);
        }
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_LAST:
        xitk_lock (gui->xitk, 1);
        if (gui->playlist.num) {
          if (xine_get_status (gui->stream) == XINE_STATUS_PLAY)
            gui_stop (NULL, gui);
          gui->playlist.cur = gui->playlist.num - 1;
          gui_current_set_index (gui, GUI_MMK_CURRENT);
          gui_play (NULL, gui);
        }
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_FLUSH:
        xitk_lock (gui->xitk, 1);
        playlist_action (gui, PLAYLIST_CLEAR);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_ADD:
        xitk_lock (gui->xitk, 1);
        gui_dndcallback (gui, (char *)shdr.data);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_NEXT:
        xitk_lock (gui->xitk, 1);
        gui_playlist_start_next (gui, 1);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_PREV:
        xitk_lock (gui->xitk, 1);
        gui_playlist_start_next (gui, -1);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_LOAD:
        xitk_lock (gui->xitk, 1);
        gui_playlist_add_item (gui, (const char *)shdr.data, 1, GUI_ITEM_TYPE_PLAYLIST, 1);
        gui_current_set_index (gui, GUI_MMK_CURRENT);
        playlist_update_playlist (gui);
        if (gui->playlist.num > 0)
          panel_playback_ctrl (gui->panel, 1);
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_STOP:
        xitk_lock (gui->xitk, 1);
        if (xine_get_status (gui->stream) != XINE_STATUS_STOP) {
          gui_playlist_lock (gui);
          gui->playlist.control |= PLAYLIST_CONTROL_STOP;
          gui_playlist_unlock (gui);
        }
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_PLAYLIST_CONTINUE:
        xitk_lock (gui->xitk, 1);
        if (xine_get_status (gui->stream) != XINE_STATUS_STOP) {
          gui_playlist_lock (gui);
          gui->playlist.control &= ~PLAYLIST_CONTROL_STOP;
          gui_playlist_unlock (gui);
        }
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_VOLUME:
        xitk_lock (gui->xitk, 1);
        {
          int *vol = (int *)shdr.data;
          if ((gui->mixer.type_volume == SOUND_CARD_MIXER) &&
            (gui->mixer.level[SOUND_CARD_MIXER] >= 0) && ((*vol >= 0) && (*vol <= 100)))
            gui_set_audio_vol (gui, *vol);
          else if ((gui->mixer.type_volume == SOFTWARE_MIXER) && ((*vol >= 0) && (*vol <= 200)))
            gui_set_amp_level (gui, *vol);
        }
        xitk_lock (gui->xitk, 0);
        break;

      case CMD_AMP:
        {
          int *amp = (int *)shdr.data;
          if ((*amp >= 0) && (*amp <= 200)) {
            xitk_lock (gui->xitk, 1);
            gui_set_amp_level (gui, *amp);
            xitk_lock (gui->xitk, 0);
          }
        }
        break;

      case CMD_LOOP:
        {
          int *loop = (int *)shdr.data;
          switch (*loop) {
            case PLAYLIST_LOOP_NO_LOOP:
            case PLAYLIST_LOOP_LOOP:
            case PLAYLIST_LOOP_REPEAT:
            case PLAYLIST_LOOP_SHUFFLE:
            case PLAYLIST_LOOP_SHUF_PLUS:
            case PLAYLIST_LOOP_MODES_NUM:
              xitk_lock (gui->xitk, 1);
              gui->playlist.loop = *loop;
              xitk_lock (gui->xitk, 0);
              break;
          }
        }
        break;

      case CMD_GET_SPEED_STATUS:
        {
          uint32_t status = 2;
          if (xine_get_status (gui->stream) == XINE_STATUS_PLAY) {
            int speed = xine_get_param (gui->stream, XINE_PARAM_SPEED);
            switch (speed) {
              case XINE_SPEED_NORMAL: status = 3; break;
              case XINE_SPEED_PAUSE:  status = 4; break;
              case XINE_SPEED_SLOW_4: status = 5; break;
              case XINE_SPEED_SLOW_2: status = 6; break;
              case XINE_SPEED_FAST_2: status = 7; break;
              case XINE_SPEED_FAST_4: status = 8; break;
              default: /* Dude! */    status = 1;
            }
          }
          send_packet (shdr.fd, CMD_GET_SPEED_STATUS, &status, sizeof (status));
        }
        break;

      case CMD_GET_TIME_STATUS_IN_SECS:
        xitk_lock (gui->xitk, 1);
        {
          uint32_t status = gui->seek.pos_time_length[1] / 1000;
          send_packet (shdr.fd, CMD_GET_TIME_STATUS_IN_SECS, &status, sizeof(status));
        }
        xitk_lock (gui->xitk, 0);
      break;

      case CMD_GET_TIME_STATUS_IN_POS:
        xitk_lock (gui->xitk, 1);
        {
          uint32_t status = gui->seek.pos_time_length[0];
          xitk_lock (gui->xitk, 0);
          send_packet (shdr.fd, CMD_GET_TIME_STATUS_IN_POS, &status, sizeof(status));
        }
        break;

      case CMD_GET_VERSION:
        send_packet (shdr.fd, CMD_GET_VERSION, VERSION,  strlen(VERSION));
        break;

      case CMD_PING:
        break;

      default:
        if (gui->verbosity >= 1)
          printf ("gui.session.unknown_command (%d).\n", shdr.hdr.command);
        ack = 0;
    }

    if (ack) {
      ctrl_header_packet_t hdr = {
        .version     = CTRL_PROTO_VERSION,
        .command     = 0,
        .data_length = 0
      };

      _send_packet (shdr.fd, NULL, &hdr);
    }
    close (shdr.fd);
    shdr.fd = -1;
    SAFE_FREE (shdr.data);
    pthread_mutex_lock (&s->mutex);
  }
  pthread_mutex_unlock (&s->mutex);

  return NULL;
}

void deinit_session (xui_session_t **s) {
  if (s && *s) {
    struct stat sstat;

    if ((*s)->gui->verbosity >= 2)
      printf ("gui.session.delete (%d, %s).\n", (*s)->ctrl_fd, (*s)->socket_name);
    pthread_mutex_lock (&(*s)->mutex);
    (*s)->going = 0;
    pthread_mutex_unlock (&(*s)->mutex);
    pthread_join ((*s)->thread_server, NULL);
    close ((*s)->ctrl_fd);
    if (((stat ((*s)->socket_name, &sstat)) > -1) && (S_ISSOCK (sstat.st_mode)))
      unlink ((*s)->socket_name);
    pthread_mutex_destroy (&(*s)->mutex);
    free ((*s)->socket_name);
    *s = NULL;
  }
}

xui_session_t *init_session (gGui_t *gui) {
  xui_session_t *s;
  union {
    struct sockaddr_un un;
    struct sockaddr sa;
  } saddr;
  int i;

  s = calloc (1, sizeof (*s));
  if (!s)
    return NULL;

  s->gui = gui;
  s->ctrl_fd = xine_socket_cloexec (AF_UNIX, SOCK_STREAM, 0);
  if (s->ctrl_fd == -1) {
    if (gui->verbosity >= 1) {
      int e = errno;
      printf ("gui.session.ctrlsocket.new.failed (%d, %s).\n", e, strerror (e));
    }
    free (s);
    return NULL;
  }

  for (i = 0; ; i++) {
    saddr.un.sun_family = AF_UNIX;
    snprintf (saddr.un.sun_path, sizeof (saddr.un.sun_path), "%s%s%d", xine_get_homedir (), "/.xine/session.", i);
    if (is_remote_running (i))
      continue;
    if (unlink (saddr.un.sun_path) == -1) {
      int e = errno;
      if ((e != ENOENT) && (gui->verbosity >= 1))
        printf ("gui.session.ctrlsocket.delete (%s, %d, %s).\n", saddr.un.sun_path, e, strerror (e));
    }
#ifdef SUN_LEN
    if ((bind (s->ctrl_fd, &saddr.sa, SUN_LEN (&saddr.un))) != -1)
#else
    if ((bind (s->ctrl_fd, &saddr.sa, sizeof (saddr.un))) != -1)
#endif
    {
      s->session_id = i;
      listen (s->ctrl_fd, 100);
      s->going = 1;

      pthread_mutex_init (&s->mutex, NULL);
      pthread_create (&s->thread_server, NULL, ctrlsocket_func, s);
      s->socket_name = strdup (saddr.un.sun_path);
      if (gui->verbosity >= 2)
        printf ("gui.session.new (%d, %s).\n", s->ctrl_fd, s->socket_name);
      return s;
    } else {
      if (gui->verbosity >= 1) {
        int e = errno;
        printf ("gui.session.ctrlsocket.bind.failed (%s, %d, %s).\n", saddr.un.sun_path, e, strerror (e));
      }
      break;
    }
  }

  if (s->ctrl_fd != -1)
    close (s->ctrl_fd);
  free (s);
  return NULL;
}

int session_handle_subopt(char *suboptarg, const char *enqueue_mrl, int *session) {
  int          playlist_first, playlist_last, playlist_clear, playlist_next, playlist_prev, playlist_stop_cont;
  int          audio_next, audio_prev, spu_next, spu_prev;
  int          volume, amp, loop, speed_status, time_status;
  int          s, c;
  int          i;
  uint32_t     state;
  char        *optstr;
  char       **mrls          = NULL;
  int          num_mrls      = 0;
  int          retval        = 0;
  char        *sopts         = suboptarg;
  int          optsess       = -1;
  char        *playlist_load = NULL;
  static char *const tokens[]= {
    /* Don't change order */
    "play",  "slow4",  "slow2",   "pause",      "fast2",
    "fast4", "stop",   "quit",    "fullscreen", "eject",
    "audio", "spu",    "session", "mrl",        "playlist",
    "pl",    "volume", "amp",     "loop",       "get_speed",
    "get_time",
    NULL
  };

  playlist_first = playlist_last = playlist_clear = playlist_next = playlist_prev = playlist_stop_cont = 0;
  audio_next     = audio_prev = spu_next = spu_prev = 0;
  volume = amp   = -1;
  state          = 0;
  loop           = -1;
  speed_status   = time_status = 0;

  while((c = getsubopt(&sopts, tokens, &optstr)) != -1) {
    uint32_t flag = 0x80000000 >> c;
    /* required arg for [0] 0000 0000 0011 1111 1110 0 [31]. */
    if ((flag & 0x003fe000) && !optstr)
        continue;
    state |= flag;

    switch(c) {

      /* audio */
    case 10:
      if(!strcasecmp (optstr, "next"))
	audio_next++;
      else if (!strcasecmp (optstr, "prev"))
	audio_prev++;
      break;

      /* spu */
    case 11:
      if (!strcasecmp (optstr, "next"))
	spu_next++;
      else if (!strcasecmp (optstr, "prev"))
	spu_prev++;
      break;

      /* session */
    case 12:
      if ((i = atoi (optstr)) >= 0)
	optsess = i;
      break;

      /* mrl */
    case 13:
      mrls = (char **) realloc (mrls, sizeof (char *) * (num_mrls + 2));
      mrls[num_mrls++] = strdup (optstr);
      mrls[num_mrls]   = NULL;
      break;

      /* playlist */
    case 14:
    case 15:
      if (!strcasecmp (optstr, "first"))
	playlist_first = 1;
      else if (!strcasecmp (optstr, "last"))
	playlist_last = 1;
      else if (!strcasecmp (optstr, "clear"))
	playlist_clear = 1;
      else if (!strcasecmp (optstr, "next"))
	playlist_next++;
      else if (!strcasecmp (optstr, "prev"))
	playlist_prev++;
      else if (!strncasecmp (optstr, "load:", 5))
	playlist_load = strdup(optstr + 5);
      else if (!strcasecmp (optstr, "stop"))
	playlist_stop_cont = -1;
      else if (!strcasecmp (optstr, "cont"))
	playlist_stop_cont = 1;
      break;

      /* volume */
    case 16:
      volume = strtol (optstr, &optstr, 10);
      break;

      /* amplification */
    case 17:
      amp = strtol (optstr, &optstr, 10);
      break;

      /* loop */
    case 18:
      if (!strcasecmp (optstr, "none"))
	loop = PLAYLIST_LOOP_NO_LOOP;
      else if (!strcasecmp (optstr, "loop"))
	loop = PLAYLIST_LOOP_LOOP;
      else if (!strcasecmp (optstr, "repeat"))
	loop = PLAYLIST_LOOP_REPEAT;
      else if (!strcasecmp (optstr, "shuffle"))
	loop = PLAYLIST_LOOP_SHUFFLE;
      else if (!strcasecmp (optstr, "shuffle+"))
	loop = PLAYLIST_LOOP_SHUF_PLUS;
      break;

      /* speed status */
    case 19:
      speed_status = 1;
      break;

    case 20:
      time_status = 1;
      if (!optstr)
        break;
      if (!strcasecmp (optstr, "p") || !strcasecmp (optstr, "pos"))
        time_status = 2;
      break;

    }
  }

  if (enqueue_mrl != NULL) {
    mrls = (char **) realloc(mrls, sizeof(char *) * (num_mrls + 2));
    mrls[num_mrls++] = strdup(enqueue_mrl);
    mrls[num_mrls]   = NULL;
  }

  *session = (optsess >= 0) ? optsess : 0;

  if(is_remote_running(*session)) {

    if (state & (0x80000000 >> (CMD_QUIT - 1)))
      remote_cmd(*session, CMD_QUIT);

    if(playlist_clear)
      remote_cmd(*session, CMD_PLAYLIST_FLUSH);

    if(playlist_load) {
      send_string(*session, CMD_PLAYLIST_LOAD, playlist_load);
      free(playlist_load);
    }

    while(playlist_next) {
      remote_cmd(*session, CMD_PLAYLIST_NEXT);
      playlist_next--;
    }

    while(playlist_prev) {
      remote_cmd(*session, CMD_PLAYLIST_PREV);
      playlist_prev--;
    }

    if(playlist_stop_cont < 0)
      remote_cmd(*session, CMD_PLAYLIST_STOP);

    if(playlist_stop_cont > 0)
      remote_cmd(*session, CMD_PLAYLIST_CONTINUE);

    for (i = 0; i < num_mrls; i++)
      send_string (*session, CMD_PLAYLIST_ADD, mrls[i]);

    /*
     * CMD_PLAY, CMD_SLOW_4, CMD_SLOW_2, CMD_PAUSE, CMD_FAST_2,
     * CMD_FAST_4, CMD_STOP, CMD_FULLSCREEN, CMD_EJECT
     */
    if (state & 0xff800000) {
      for (s = 0; s < 9; s++) {
        if ((state & (0x80000000 >> s)) && (s + 1 != CMD_QUIT))
	  remote_cmd(*session, (s + 1));
      }
    }

    while(audio_next) {
      remote_cmd(*session, CMD_AUDIO_NEXT);
      audio_next--;
    }

    while(audio_prev) {
      remote_cmd(*session, CMD_AUDIO_PREV);
      audio_prev--;
    }

    while(spu_next) {
      remote_cmd(*session, CMD_SPU_NEXT);
      spu_next--;
    }

    while(spu_prev) {
      remote_cmd(*session, CMD_SPU_PREV);
      spu_prev--;
    }

    if(volume >= 0)
      send_uint32(*session, CMD_VOLUME, (uint32_t) volume);

    if(amp >= 0)
      send_uint32(*session, CMD_AMP, (uint32_t) amp);

    if(loop > -1)
      send_uint32(*session, CMD_LOOP, (uint32_t) loop);

    if(playlist_first)
      remote_cmd(*session, CMD_PLAYLIST_FIRST);

    if(playlist_last)
      remote_cmd(*session, CMD_PLAYLIST_LAST);

    if(speed_status)
      retval = get_uint32(*session, CMD_GET_SPEED_STATUS);

    if(time_status) {
      switch(time_status) {
      case 1:
	retval = get_uint32(*session, CMD_GET_TIME_STATUS_IN_SECS);
	break;

      case 2:
	retval = get_uint32(*session, CMD_GET_TIME_STATUS_IN_POS);
	break;
      }

      printf ("%d\n", retval);
    }

  }
  else
    printf (_("Session %d isn't running.\n"), *session);

  for(i = 0; i < num_mrls; i++)
    free(mrls[i]);

  free(mrls);

  return retval;
}

