
/* 
 * International Color Consortium color transform expanded support
 *
 * Author:  Graeme W. Gill
 * Date:    23/22/2021
 * Version: 1.00
 *
 * Copyright 2011-2021 Graeme W. Gill
 * All rights reserved.
 * This material is licenced under the GNU AFFERO GENERAL PUBLIC LICENSE Version 3 :-
 * see the License.txt file for licencing details.
 *
 */

/*
 * Code for the cicam02 blue linearization hack.
 *
 * Inspired by "Color gamut mapping in a hue-linearized CIELAB color space"
 * by Gustav Braun and Mark Fairchild.
 */


#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include "aconfig.h"
#include "icc.h"
#include "numlib.h"
#include "plot.h"
#include "xcam.h"
#include "cam02.h"
//#include "xicc.h"
#include "ui.h"

#define BLUELIN_h0    210.0     /* Low limit of hue angle to tweak */
#define BLUELIN_h1    330.0     /* High limit of hue angle at C0 to tweak */
#define BLUELIN_C0    50.0      /* Low limit of Chroma to tweak */
#define BLUELIN_C10   80.0      /* High end of Chroma to tweak */
#define BLUELIN_C11   140.0     /* High end of Hue and Chroma to tweak */
#define BLUELIN_AMNT  0.60       /* Amount to tweak hue (< 1.0 increases tweak) */

/* We are using simple linear interpolation for this. */
static void bluelin_fwd(double out[3], double in[3]) {
	double J, C, h;

	C = sqrt(in[1] * in[1] + in[2] * in[2]);
    h = (180.0/DBL_PI) * atan2(in[2], in[1]);
	h = (h < 0.0) ? h + 360.0 : h;

	if (h >= BLUELIN_h0 && h <= BLUELIN_h1
	 && C > BLUELIN_C0) {
		double hh = (h - BLUELIN_h0)/(BLUELIN_h1 - BLUELIN_h0);
		double gr, c1, amnt;

		c1 = (1.0 - hh) * BLUELIN_C10 + hh * BLUELIN_C11;
	
		gr = (C - BLUELIN_C0)/(c1 - BLUELIN_C0);
		if (gr < 0.0)
			gr = 0.0;
		else if (gr > 1.0)
			gr = 1.0;
	
		amnt = (1.0 - gr) * 1.0 + gr * BLUELIN_AMNT;

		if (hh < 0.5) {
			hh = hh * amnt;
		} else {
			hh = (0.5 * amnt) + (hh - 0.5) * (1.0 - 0.5 * amnt)/0.5;
		}

		h = BLUELIN_h0 + hh * (BLUELIN_h1 - BLUELIN_h0);
	}

	h = DBL_PI/180.0 * h;
	out[0] = in[0];
	out[1] = C * cos(h);
	out[2] = C * sin(h);
}

static void bluelin_bwd(double out[3], double in[3]) {
	double J, C, h;

	C = sqrt(in[1] * in[1] + in[2] * in[2]);
    h = (180.0/DBL_PI) * atan2(in[2], in[1]);
	h = (h < 0.0) ? h + 360.0 : h;

	if (h >= BLUELIN_h0 && h <= BLUELIN_h1
	 && C > BLUELIN_C0) {
		double hh = (h - BLUELIN_h0)/(BLUELIN_h1 - BLUELIN_h0);
		double ho = hh;
		double gr, c1 = 0.0, pc1 = -100.0, amnt;
		int j;

		/* Should be able to compute inverse analytically, but for the moment */
		/* we'll simply itterate to sufficient accuracy... */
		for (j = 0; fabs(c1 - pc1) > 0.02 && j < 20; j++) {
			pc1 = c1;
			c1 = (1.0 - ho) * BLUELIN_C10 + ho * BLUELIN_C11;
	
			gr = (C - BLUELIN_C0)/(c1 - BLUELIN_C0);
			if (gr < 0.0)
				gr = 0.0;
			else if (gr > 1.0)
				gr = 1.0;
		
			amnt = (1.0 - gr) * 1.0 + gr * BLUELIN_AMNT;
	
			if (hh < (0.5 * amnt)) {
				ho = hh / amnt;
			} else {
				ho = 0.5 + (hh - 0.5 * amnt) * (1.0 - 0.5)/(1.0 - 0.5 * amnt);
			}
		}
		h = BLUELIN_h0 + ho * (BLUELIN_h1 - BLUELIN_h0);
	}

	h = DBL_PI/180.0 * h;
	out[0] = in[0];
	out[1] = C * cos(h);
	out[2] = C * sin(h);
}

typedef double xyzarray[3];

xyzarray bandh[];
xyzarray eandf[];

int
main(void) {
	double in[3], lin[3], out[3];
	double C, h;
	cam02 *cam;
	plot_g _g, *g = &_g;
	double x1, y1;
	double maxde = 0.0;
	int hstart = 0;
	int i, j;
	float black[3] = { 0.0, 0.0, 0.0 };

	init_g(g);

	/* Explore effect and check it inverts */

	for (h = 0.0; h <= 360.0; h += 10.0) {
		for (C = 0.0; C <= 150.0; C += 2.0) {
			double de;

			in[0] = 50.0;
			in[1] = C;
			in[2] = h;
			icmLCh2Lab(in, in);

			bluelin_fwd(lin, in);
			bluelin_bwd(out, lin);

			icmLab2LCh(in, in);
			icmLab2LCh(lin, lin);
			icmLab2LCh(out, out);

			/* We're plotting hue, Chromaticity */
			if (C > 0.0) {
				add_vec_g(g, x1, y1, lin[2], lin[1], NULL);
				x1 = lin[2];
				y1 = lin[1];
			} else {
				x1 = h;
				y1 = C;
			}

			de = icmNorm33(in, out);
			if (de > maxde)
				maxde = de;

			if (de > 0.01) {
				printf("%f %f %f -> %f %f %f de %f\n", in[0], in[1], in[2], out[0], out[1], out[2], de);
			}
		}
	}

	printf("Maxde = %f\n",maxde);
	printf("Hue transformation:\n");
	do_plot_g(g, 0.0, 0.0, 0.0, 0.0, 1.0, 0, 1);
	clear_g(g);

	/* Plot the constant hue data */
	cam = new_cam02();
	/* Data has D65 white point */
	cam->set_view(cam, vc_average, icmD65_ary3, 33, 0.2, 0.0, 0.0, 0.0, icmD65_ary3, 0, 0.0, 0.0, NULL);
	cam->bluelin = 0;		/* Turn off built in version */

#ifndef NEVER
	hstart = 0;
	for (i = 0; ; i++) {
		double xyz[3], Jab[3];

		if (bandh[i][1] < 0.0) 
			break;

		if (bandh[i][1] == 0.0) {
			hstart = 1;
		}

		icmScale3(xyz, bandh[i], 0.01);
		cam->XYZ_to_cam(cam, Jab, xyz);
		bluelin_fwd(Jab, Jab);
		if (hstart) {
			x1 = Jab[1];
			y1 = Jab[2];
			hstart = 0;
		} else {
			add_vec_g(g, x1, y1, Jab[1], Jab[2], NULL);
			x1 = Jab[1];
			y1 = Jab[2];
		}
//			add_sym_g(g, Jab[1], Jab[2], plotDiamond, black, NULL);
	}

	printf("Berns and Hung::\n");
	do_plot_g(g, 0.0, 0.0, 0.0, 0.0, 1.0, 0, 1);
	clear_g(g);
#endif // NEVER


#ifndef NEVER
	hstart = 0;
	for (i = 0; ; i++) {
		double xyz[3], Jab[3];

		if (eandf[i][1] < 0.0) 
			break;

		if (eandf[i][1] == 0.0) {
			hstart = 1;
		}

		icmScale3(xyz, eandf[i], 0.01);
		cam->XYZ_to_cam(cam, Jab, xyz);
		bluelin_fwd(Jab, Jab);
		if (hstart) {
			x1 = Jab[1];
			y1 = Jab[2];
			hstart = 0;
		} else {
			add_vec_g(g, x1, y1, Jab[1], Jab[2], NULL);
			x1 = Jab[1];
			y1 = Jab[2];
		}
	}

	printf("Ebner and Fairchil:\n");
	do_plot_g(g, 0.0, 0.0, 0.0, 0.0, 1.0, 0, 1);
	clear_g(g);
#endif

	return 0;
}

/* Constant hue reference data */
/* 0,0,0 marks start of a hue */
/* -1,-1,-1 marks end of data */

/* Berns and Hung (1995) data set, table IV, variable luminance */
xyzarray bandh[] = {
	0,0,0,
	4.74,2.99,0.48,
	10.36,6.24,0.95,
	19.13,11.25,1.19,
	32.06,18.42,2.00,
	49.21,28.12,2.58,
	59.12,40.75,14.70,
	68.39,56.68,35.70,
	82.47,76.30,68.26,
	54.45,30.90,2.54,
	0,0,0,
	3.48,2.99,0.85,
	7.24,6.24,1.49,
	12.97,11.25,2.34,
	20.97,18.42,3.40,
	31.96,28.12,4.78,
	45.10,40.75,5.83,
	62.88,56.68,7.85,
	77.59,76.30,41.69,
	68.22,61.38,7.91,
	0,0,0,
	3.04,2.99,0.83,
	6.24,6.24,1.41,
	10.62,11.25,2.05,
	16.55,18.42,3.00,
	25.72,28.12,4.52,
	36.51,40.75,6.30,
	49.80,56.68,8.62,
	65.79,76.30,11.49,
	86.15,100.41,14.92,
	0,0,0,
	2.55,2.99,0.60,
	5.07,6.24,1.11,
	8.99,11.25,1.82,
	14.15,18.42,3.04,
	20.85,28.12,4.57,
	29.27,40.75,6.77,
	40.71,56.68,9.19,
	53.86,76.30,12.42,
	65.62,94.72,15.28,
	0,0,0,
	1.81,2.99,0.68,
	3.19,6.24,1.44,
	5.99,11.25,2.77,
	9.31,18.42,4.10,
	14.29,28.12,6.38,
	20.22,40.75,8.20,
	27.52,56.68,10.67,
	36.43,76.30,14.41,
	35.93,77.65,13.93,
	0,0,0,
	2.03,2.99,4.54,
	4.21,6.24,9.44,
	7.36,11.25,15.44,
	11.98,18.42,25.00,
	17.52,28.12,33.65,
	24.98,40.75,45.66,
	34.39,56.68,61.34,
	45.77,76.30,79.24,
	50.38,84.76,86.02,
	0,0,0,
	2.64,2.99,6.14,
	5.18,6.24,12.38,
	8.99,11.25,21.68,
	14.05,18.42,34.08,
	21.06,28.12,51.05,
	29.14,40.75,69.47,
	39.60,56.68,93.01,
	56.89,76.30,116.12,
	49.55,72.27,114.91,
	0,0,0,
	2.87,2.99,7.64,
	5.88,6.24,16.11,
	10.23,11.25,28.98,
	15.91,18.42,46.23,
	24.24,28.12,70.16,
	33.44,40.75,98.09,
	48.22,56.68,108.94,
	70.64,76.30,113.17,
	37.25,45.26,111.12,
	0,0,0,
	5.20,2.99,23.94,
	11.35,6.24,54.22,
	20.31,11.25,96.15,
	22.80,18.42,92.90,
	30.05,28.12,101.82,
	39.46,40.75,105.02,
	53.22,56.68,107.06,
	71.87,76.30,109.79,
	21.68,10.77,105.09,
	0,0,0,
	5.27,2.99,8.50,
	11.40,6.24,21.73,
	20.97,11.25,43.13,
	34.75,18.42,78.57,
	47.15,28.12,107.15,
	55.18,40.75,109.58,
	66.66,56.68,111.96,
	79.84,76.30,115.40,
	44.79,23.30,105.63,
	0,0,0,
	5.26,2.99,5.82,
	11.20,6.24,14.50,
	20.53,11.25,27.51,
	33.79,18.42,45.88,
	51.91,28.12,75.26,
	70.08,40.75,105.76,
	76.29,56.68,110.69,
	85.39,76.30,114.39,
	68.73,36.83,106.85,
	0,0,0,
	11.06,6.24,10.47,
	19.96,11.25,16.62,
	32.72,18.42,25.91,
	50.38,28.12,40.77,
	68.29,40.75,65.68,
	77.47,56.68,86.26,
	88.50,76.30,104.63,
	64.36,35.53,53.66,
	65.64,35.53,45.43,
	-1,-1,-1
};

/* Ebner and Fairchild (1998) set */
xyzarray eandf[] = {
0,0,0, 2.4951, 1.9086, 2.0329, 3.3296, 1.9086, 2.1063, 5.7386, 2.9891, 3.2447, 7.3706, 6.2359, 7.0514, 9.0418, 6.2359, 6.8901, 10.9443, 6.2359, 6.7881, 12.0597, 6.2359, 6.6805, 21.6475, 11.251, 12.3602, 20.4172, 18.4187, 20.1965, 23.6417, 18.4187, 20.1739, 27.1865, 18.4187, 19.5653, 31.0729, 18.4187, 19.715, 35.5801, 18.4187, 18.1154, 52.2309, 28.1233, 30.6758, 43.6118, 40.7494, 44.221, 48.8989, 40.7494, 44.8546, 54.6008, 40.7494, 44.2074, 59.4653, 40.7494, 45.0086, 69.1717, 56.6813, 61.9143, 80.3849, 76.3034, 83.6166, 
0,0,0, 2.3295, 2.9891, 1.2151, 5.0776, 6.2359, 3.235, 4.5813, 6.2359, 1.0648, 12.3217, 14.5417, 9.1954, 11.1245, 14.5417, 4.5894, 13.1275, 18.4187, 2.587, 24.2342, 28.1233, 20.1576, 22.2813, 28.1233, 12.0243, 20.4891, 28.1233, 6.415, 20.0979, 28.1233, 4.006, 42.0356, 48.2781, 37.6153, 39.2198, 48.2781, 24.84, 36.5407, 48.2781, 15.3761, 34.4596, 48.2781, 8.4394, 40.0128, 56.6813, 8.182, 52.6749, 76.3034, 11.314, 67.2532, 76.3034, 62.5084, 62.8152, 76.3034, 44.9196, 58.6004, 76.3034, 30.9764, 55.5877, 76.3034, 19.4968, 
0,0,0, 2.057, 2.9891, 1.6832, 1.3952, 2.9891, 0.8032, 8.6184, 11.251, 8.5273, 6.9267, 11.251, 5.3448, 5.5532, 11.251, 2.8928, 8.2777, 18.4187, 4.1966, 22.8649, 28.1233, 23.4216, 19.4735, 28.1233, 17.196, 16.3555, 28.1233, 12.4344, 13.8615, 28.1233, 7.9293, 12.6312, 28.1233, 6.3504, 28.1591, 56.6813, 16.2474, 25.2804, 56.6813, 12.9911, 47.6451, 56.6813, 49.9746, 41.9222, 56.6813, 39.8854, 37.2596, 56.6813, 29.3883, 32.1493, 56.6813, 23.1624, 29.5765, 66.007, 14.7727, 47.6799, 76.3034, 35.121, 70.119, 87.6183, 69.0486, 
0,0,0, 2.108, 2.9891, 2.7904, 1.6702, 2.9891, 2.3311, 3.4115, 6.2359, 5.1996, 8.8419, 11.251, 11.3399, 7.2572, 11.251, 9.8606, 6.1947, 11.251, 9.3344, 10.0632, 18.4187, 15.1057, 23.2833, 28.1233, 28.5625, 20.1319, 28.1233, 26.9643, 17.3352, 28.1233, 24.4685, 15.2558, 28.1233, 23.2349, 22.1968, 40.7494, 33.9779, 48.3497, 56.6813, 57.9764, 43.2329, 56.6813, 54.4056, 38.5515, 56.6813, 50.1932, 34.1577, 56.6813, 47.0353, 30.8994, 56.6813, 44.527, 35.9901, 66.007, 50.7477, 49.0016, 76.3034, 63.8159, 62.2486, 82.967, 76.3862, 
0,0,0, 2.3453, 2.9891, 3.7174, 1.926, 2.9891, 4.4245, 4.0173, 6.2359, 9.6191, 9.4684, 11.251, 13.3777, 8.3988, 11.251, 15.2001, 7.3638, 11.251, 16.5584, 11.6981, 18.4187, 25.5249, 24.3887, 28.1233, 31.9199, 22.2844, 28.1233, 34.5849, 20.2715, 28.1233, 36.817, 18.2763, 28.1233, 37.3909, 25.4976, 40.7494, 52.8133, 50.1257, 56.6813, 64.0076, 46.5487, 56.6813, 66.0339, 43.1626, 56.6813, 68.3981, 39.8701, 56.6813, 69.2512, 35.2616, 56.6813, 70.4797, 46.08, 74.1641, 88.1726, 58.2443, 80.7044, 93.9569, 71.4332, 87.6183, 99.9575, 
0,0,0, 2.4666, 2.9891, 4.4851, 2.2448, 2.9891, 5.9111, 4.6165, 6.2359, 12.0332, 9.74, 11.251, 14.9949, 8.8882, 11.251, 18.3141, 8.1128, 11.251, 21.4127, 13.2261, 18.4187, 34.5513, 24.8999, 28.1233, 35.3653, 23.1907, 28.1233, 40.724, 21.6271, 28.1233, 46.921, 19.8433, 28.1233, 50.4344, 28.2041, 40.7494, 69.0229, 50.8535, 56.6813, 68.8802, 47.8889, 56.6813, 76.2798, 44.6584, 56.6813, 82.046, 41.6195, 56.6813, 88.1993, 39.0704, 56.6813, 94.4482, 49.8666, 66.007, 100.2007, 62.9735, 76.3034, 103.0157, 78.0945, 87.6183, 105.3757, 
0,0,0, 2.4364, 1.9086, 1.4688, 3.168, 1.9086, 0.9391, 5.5818, 2.9891, 1.1895, 7.265, 6.2359, 5.462, 8.7473, 6.2359, 4.1462, 10.3312, 6.2359, 2.8546, 11.6178, 6.2359, 2.127, 20.8807, 11.251, 3.7728, 20.1146, 18.4187, 16.7921, 23.0654, 18.4187, 14.2768, 26.3119, 18.4187, 12.1097, 29.4922, 18.4187, 9.1904, 33.7924, 18.4187, 5.6569, 41.9121, 22.9298, 6.3588, 52.713, 34.0472, 18.642, 43.1587, 40.7494, 39.0086, 47.8373, 40.7494, 33.728, 52.8667, 40.7494, 29.0411, 66.526, 56.6813, 45.7942, 79.0894, 76.3034, 74.2766, 
0,0,0, 2.6197, 2.9891, 4.9052, 2.4789, 2.9891, 6.9067, 5.0943, 6.2359, 14.577, 10.1162, 11.251, 15.9858, 9.6673, 11.251, 20.6324, 9.2055, 11.251, 26.0341, 14.8942, 18.4187, 42.8274, 25.5352, 28.1233, 37.1142, 24.4632, 28.1233, 44.6614, 23.4074, 28.1233, 53.131, 22.2932, 28.1233, 62.3583, 31.9264, 40.7494, 92.1437, 44.1905, 48.2781, 61.8135, 42.3672, 48.2781, 71.7357, 40.7141, 48.2781, 82.9579, 39.2461, 48.2781, 95.6828, 47.2962, 56.6813, 100.3071, 57.2536, 66.007, 101.5921, 67.9355, 76.3034, 103.436, 
0,0,0, 2.7982, 2.9891, 8.6685, 4.1208, 4.4155, 7.1309, 4.1612, 4.4155, 10.1433, 6.0966, 6.2359, 18.7302, 10.5606, 11.251, 16.4317, 10.4042, 11.251, 21.4695, 10.4747, 11.251, 27.5383, 10.5836, 11.251, 33.8709, 18.1547, 18.4187, 55.3577, 26.2553, 28.1233, 38.0059, 25.8303, 28.1233, 46.5439, 25.9521, 28.1233, 56.6217, 26.7324, 28.1233, 68.0523, 28.0301, 28.1233, 84.418, 33.806, 34.0472, 96.8792, 42.4315, 45.1644, 99.9617, 52.8874, 56.6813, 73.1468, 52.3387, 56.6813, 86.2998, 52.4765, 56.6813, 101.3278, 70.8398, 76.3034, 102.9002, 
0,0,0, 3.0398, 2.9891, 6.1233, 3.3574, 2.9891, 10.1548, 3.7776, 2.9891, 14.0607, 8.1239, 6.2359, 28.9466, 8.3832, 8.4984, 14.7224, 8.87, 8.4984, 21.8435, 9.7338, 8.4984, 30.3378, 11.1523, 8.4984, 39.3449, 15.0163, 11.251, 50.4059, 17.9346, 18.4187, 28.9351, 18.4058, 18.4187, 40.1114, 19.842, 18.4187, 52.7573, 21.8326, 18.4187, 66.4797, 24.474, 18.4187, 83.4294, 32.3931, 28.1233, 95.1349, 32.5501, 34.0472, 50.2969, 33.1968, 34.0472, 66.2406, 35.3672, 34.0472, 83.913, 36.8082, 34.0472, 96.9789, 45.6537, 45.1644, 100.359, 53.8261, 56.6813, 80.0104, 55.3271, 56.6813, 101.2963, 72.3061, 76.3034, 105.1867, 
0,0,0, 3.7048, 2.9891, 5.9648, 4.754, 2.9891, 9.7663, 6.0374, 2.9891, 14.6405, 7.3117, 2.9891, 19.7734, 15.1352, 6.2359, 42.7145, 19.0702, 8.4984, 52.0012, 9.6873, 8.4984, 14.6268, 11.4333, 8.4984, 22.0239, 13.6517, 8.4984, 30.3894, 16.1808, 8.4984, 40.3975, 27.2723, 11.251, 78.181, 34.004, 18.4187, 85.7156, 19.9887, 18.4187, 29.2578, 22.6043, 18.4187, 41.2528, 26.022, 18.4187, 53.9744, 29.6726, 18.4187, 69.3691, 42.7495, 28.1233, 95.3079, 35.8782, 34.0472, 51.104, 39.7286, 34.0472, 68.1086, 44.5219, 34.0472, 86.2127, 52.0133, 40.7494, 99.9868, 58.4479, 56.6813, 81.8948, 63.3425, 56.6813, 104.0201, 77.4496, 76.3034, 106.882, 
0,0,0, 5.9792, 2.9891, 6.7983, 5.6611, 4.4155, 6.4892, 7.4487, 4.4155, 8.4184, 12.5371, 6.2359, 14.8021, 16.887, 14.5417, 19.8849, 20.5757, 14.5417, 23.2471, 24.6615, 14.5417, 27.6573, 29.0646, 14.5417, 33.9618, 37.1009, 18.4187, 44.326, 55.3607, 28.1233, 75.2024, 31.4203, 28.1233, 36.8351, 36.8426, 28.1233, 42.4991, 42.7739, 28.1233, 49.2386, 48.8039, 28.1233, 60.1262, 65.0875, 40.7494, 82.7335, 52.4811, 48.2781, 61.791, 59.8114, 48.2781, 71.3579, 67.3205, 48.2781, 84.8059, 72.5461, 56.6813, 91.3143, 82.5715, 76.3034, 98.8142, 
0,0,0, 3.6781, 2.9891, 1.4811, 4.3912, 2.9891, 0.4572, 7.2096, 6.2359, 3.6436, 8.6369, 6.2359, 1.6378, 9.3726, 6.2359, 0.7023, 17.1277, 11.251, 1.2299, 15.9324, 14.5417, 9.8473, 18.4248, 14.5417, 5.7809, 20.8121, 14.5417, 2.8115, 27.2709, 18.4187, 1.7329, 30.0809, 28.1233, 21.2298, 33.7425, 28.1233, 14.0395, 37.1325, 28.1233, 8.2291, 38.9601, 28.1233, 3.5885, 40.9676, 28.1233, 2.275, 52.976, 40.7494, 11.72, 50.1955, 48.2781, 38.0164, 54.5519, 48.2781, 26.1853, 63.3032, 56.6813, 32.6901, 77.0216, 76.3034, 65.1076, 
0,0,0, 3.3197, 2.9891, 1.148, 3.5138, 2.9891, 0.6227, 6.6846, 6.2359, 3.0931, 7.4603, 6.2359, 1.1325, 13.4731, 11.251, 1.8616, 14.8686, 14.5417, 8.7303, 16.3904, 14.5417, 4.3255, 21.7903, 18.4187, 2.9308, 28.5178, 28.1233, 19.2981, 30.4615, 28.1233, 11.2259, 32.3158, 28.1233, 5.7361, 33.187, 28.1233, 4.2045, 46.0119, 40.7494, 5.6175, 48.3113, 48.2781, 35.8235, 50.4509, 48.2781, 22.8894, 52.5746, 48.2781, 13.4945, 54.6262, 48.2781, 7.0769, 61.6913, 56.6813, 16.4547, 67.8322, 66.007, 33.3101, 75.4525, 76.3034, 54.1791, 
0,0,0, 2.8909, 2.9891, 1.0142, 5.9861, 6.2359, 2.8395, 6.187, 6.2359, 1.0384, 10.8221, 11.251, 1.7582, 13.8453, 14.5417, 8.4043, 14.0725, 14.5417, 3.7708, 17.4718, 18.4187, 2.7642, 26.6956, 28.1233, 18.6195, 26.7106, 28.1233, 10.2615, 26.5222, 28.1233, 4.8764, 37.8172, 40.7494, 6.0499, 45.2991, 48.2781, 34.9471, 44.8363, 48.2781, 21.7887, 44.4752, 48.2781, 12.426, 51.2315, 56.6813, 8.5942, 70.9748, 76.3034, 58.9282, 70.1174, 76.3034, 39.8412, 69.8154, 76.3034, 25.3302, 70.5383, 76.3034, 14.8054, 70.4427, 76.3034, 11.249, 
-1,-1,-1
};


