/* RetroArch - A frontend for libretro.
 *  Copyright (C) 2011-2016 - Daniel De Matteis
 *
 * RetroArch is free software: you can redistribute it and/or modify it under the terms
 * of the GNU General Public License as published by the Free Software Found-
 * ation, either version 3 of the License, or (at your option) any later version.
 *
 * RetroArch is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with RetroArch.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdint.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <boolean.h>

#include <file/file_path.h>
#include <queues/task_queue.h>
#include <string/stdstring.h>
#include <retro_timers.h>

#include "cocoa/cocoa_common.h"
#include "cocoa/apple_platform.h"
#include "../ui_companion_driver.h"
#include "../../audio/audio_driver.h"
#include "../../configuration.h"
#include "../../frontend/frontend.h"
#include "../../input/drivers/cocoa_input.h"
#include "../../input/input_driver.h"
#include "../../input/drivers_keyboard/keyboard_event_apple.h"
#include "../../retroarch.h"
#include "../../tasks/task_content.h"
#include "../../verbosity.h"
#include "../../core_info.h"

#if HAVE_SWIFT
#if TARGET_OS_TV
#import "RetroArchTV-Swift.h"
#else
#import "RetroArch-Swift.h"
#endif
#endif

#ifdef HAVE_MENU
#include "../../menu/menu_setting.h"
#endif

#import <AVFoundation/AVFoundation.h>
#import <CoreFoundation/CoreFoundation.h>

#import <MetricKit/MetricKit.h>
#import <MetricKit/MXMetricManager.h>

#ifdef HAVE_MFI
#import <GameController/GameController.h>
#import <GameController/GCMouse.h>
#import <GameController/GCMouseInput.h>
#endif

#ifdef HAVE_KSCRASH
#import <KSCrash.h>
#import <KSCrashConfiguration.h>
#import <KSCrashReportStore.h>
#import <KSCrashInstallation.h>
#import <KSCrashReport.h>
#endif

#ifdef HAVE_SDL2
#define SDL_MAIN_HANDLED
#include "SDL.h"
#endif

#if defined(HAVE_COCOA_METAL) || defined(HAVE_COCOATOUCH)
#import "JITSupport.h"
id<ApplePlatform> apple_platform;
#else
static id apple_platform;
#endif

static void ui_companion_cocoatouch_event_command(
      void *data, enum event_command cmd) { }

static struct string_list *ui_companion_cocoatouch_get_app_icons(void)
{
   static struct string_list *list = NULL;
   static dispatch_once_t onceToken;

   dispatch_once(&onceToken, ^{
         union string_list_elem_attr attr;
         attr.i = 0;
         NSDictionary *iconfiles = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleIcons"];
         NSString *primary;
         const char *cstr;
#if TARGET_OS_TV
         primary = iconfiles[@"CFBundlePrimaryIcon"];
#else
         primary = iconfiles[@"CFBundlePrimaryIcon"][@"CFBundleIconName"];
#endif
         list = string_list_new();
         cstr = [primary cStringUsingEncoding:kCFStringEncodingUTF8];
         if (cstr)
            string_list_append(list, cstr, attr);

         NSArray<NSString *> *alts;
#if TARGET_OS_TV
         alts = iconfiles[@"CFBundleAlternateIcons"];
#else
         alts = [iconfiles[@"CFBundleAlternateIcons"] allKeys];
#endif
         NSArray<NSString *> *sorted = [alts sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)];
         for (NSString *str in sorted)
         {
            cstr = [str cStringUsingEncoding:kCFStringEncodingUTF8];
            if (cstr)
               string_list_append(list, cstr, attr);
         }
      });

   return list;
}

static void ui_companion_cocoatouch_set_app_icon(const char *iconName)
{
   NSString *str;
   if (!string_is_equal(iconName, "Default"))
      str = [NSString stringWithCString:iconName encoding:NSUTF8StringEncoding];
   [[UIApplication sharedApplication] setAlternateIconName:str completionHandler:nil];
}

static uintptr_t ui_companion_cocoatouch_get_app_icon_texture(const char *icon)
{
   static NSMutableDictionary<NSString *, NSNumber *> *textures = nil;
   static dispatch_once_t once;
   dispatch_once(&once, ^{
      textures = [NSMutableDictionary dictionaryWithCapacity:6];
   });

   NSString *iconName = [NSString stringWithUTF8String:icon];
   if (!textures[iconName])
   {
      UIImage *img = [UIImage imageNamed:iconName];
      if (!img)
      {
         RARCH_LOG("[Cocoa] Could not load %s.\n", icon);
         return 0;
      }
      NSData *png = UIImagePNGRepresentation(img);
      if (!png)
      {
         RARCH_LOG("[Cocoa] Could not get png for %s.\n", icon);
         return 0;
      }

      uintptr_t item;
      gfx_display_reset_textures_list_buffer(&item, TEXTURE_FILTER_MIPMAP_LINEAR,
                                             (void*)[png bytes], (unsigned int)[png length], IMAGE_TYPE_PNG,
                                             NULL, NULL);
      textures[iconName] = [NSNumber numberWithUnsignedLong:item];
   }

   return [textures[iconName] unsignedLongValue];
}

void get_ios_version(int *major, int *minor)
{
   static int savedMajor, savedMinor;
   static dispatch_once_t onceToken;

   dispatch_once(&onceToken, ^ {
         NSArray *decomposed_os_version = [[UIDevice currentDevice].systemVersion componentsSeparatedByString:@"."];
         if (decomposed_os_version.count > 0)
            savedMajor = (int)[decomposed_os_version[0] integerValue];
         if (decomposed_os_version.count > 1)
            savedMinor = (int)[decomposed_os_version[1] integerValue];
      });
   if (major) *major = savedMajor;
   if (minor) *minor = savedMinor;
}

bool ios_running_on_ipad(void)
{
   return (UI_USER_INTERFACE_IDIOM() == UIUserInterfaceIdiomPad);
}

/* Input helpers: This is kept here because it needs ObjC */
static void handle_touch_event(NSArray* touches)
{
#if !TARGET_OS_TV
   unsigned i;
   cocoa_input_data_t *apple = (cocoa_input_data_t*)
      input_state_get_ptr()->current_data;
   float scale               = cocoa_screen_get_native_scale();

   if (!apple)
      return;

   apple->touch_count = 0;

   for (i = 0; i < touches.count && (apple->touch_count < MAX_TOUCHES); i++)
   {
      UITouch      *touch = [touches objectAtIndex:i];
      CGPoint       coord = [touch locationInView:[touch view]];
      if (touch.phase != UITouchPhaseEnded && touch.phase != UITouchPhaseCancelled)
      {
         apple->touches[apple->touch_count   ].screen_x = coord.x * scale;
         apple->touches[apple->touch_count ++].screen_y = coord.y * scale;
      }
   }
#endif
}

#ifndef HAVE_APPLE_STORE
/* iOS7 Keyboard support */
@interface UIEvent(iOS7Keyboard)
@property(readonly, nonatomic) long long _keyCode;
@property(readonly, nonatomic) _Bool _isKeyDown;
@property(retain, nonatomic) NSString *_privateInput;
@property(nonatomic) long long _modifierFlags;
- (struct __IOHIDEvent { }*)_hidEvent;
@end

@interface UIApplication(iOS7Keyboard)
- (void)handleKeyUIEvent:(UIEvent*)event;
- (id)_keyCommandForEvent:(UIEvent*)event;
@end
#endif

@interface RApplication : UIApplication
@end

@implementation RApplication

#ifndef HAVE_APPLE_STORE
/* Keyboard handler for iOS 7. */

/* This is copied here as it isn't
 * defined in any standard iOS header */
enum
{
   NSAlphaShiftKeyMask                  = 1 << 16,
   NSShiftKeyMask                       = 1 << 17,
   NSControlKeyMask                     = 1 << 18,
   NSAlternateKeyMask                   = 1 << 19,
   NSCommandKeyMask                     = 1 << 20,
   NSNumericPadKeyMask                  = 1 << 21,
   NSHelpKeyMask                        = 1 << 22,
   NSFunctionKeyMask                    = 1 << 23,
   NSDeviceIndependentModifierFlagsMask = 0xffff0000U
};

/* This is specifically for iOS 9, according to the private headers */
-(void)handleKeyUIEvent:(UIEvent *)event
{
    /* This gets called twice with the same timestamp
     * for each keypress, that's fine for polling
     * but is bad for business with events. */
    static double last_time_stamp;

    if (last_time_stamp == event.timestamp)
       return [super handleKeyUIEvent:event];

    last_time_stamp        = event.timestamp;

    /* If the _hidEvent is NULL, [event _keyCode] will crash.
     * (This happens with the on screen keyboard). */
    if (event._hidEvent)
    {
        NSString       *ch = (NSString*)event._privateInput;
        uint32_t character = 0;
        uint32_t mod       = 0;
        NSUInteger mods    = event._modifierFlags;

        if (mods & NSAlphaShiftKeyMask)
           mod |= RETROKMOD_CAPSLOCK;
        if (mods & NSShiftKeyMask)
           mod |= RETROKMOD_SHIFT;
        if (mods & NSControlKeyMask)
           mod |= RETROKMOD_CTRL;
        if (mods & NSAlternateKeyMask)
           mod |= RETROKMOD_ALT;
        if (mods & NSCommandKeyMask)
           mod |= RETROKMOD_META;
        if (mods & NSNumericPadKeyMask)
           mod |= RETROKMOD_NUMLOCK;

        if (ch && ch.length != 0)
        {
            unsigned i;
            character = [ch characterAtIndex:0];

            apple_input_keyboard_event(event._isKeyDown,
                  (uint32_t)event._keyCode, 0, mod,
                  RETRO_DEVICE_KEYBOARD);

            for (i = 1; i < ch.length; i++)
                apple_input_keyboard_event(event._isKeyDown,
                      0, [ch characterAtIndex:i], mod,
                      RETRO_DEVICE_KEYBOARD);
        }

        apple_input_keyboard_event(event._isKeyDown,
              (uint32_t)event._keyCode, character, mod,
              RETRO_DEVICE_KEYBOARD);
    }

    [super handleKeyUIEvent:event];
}

/* This is for iOS versions < 9.0 */
- (id)_keyCommandForEvent:(UIEvent*)event
{
   /* This gets called twice with the same timestamp
    * for each keypress, that's fine for polling
    * but is bad for business with events. */
   static double last_time_stamp;

   if (last_time_stamp == event.timestamp)
      return [super _keyCommandForEvent:event];
   last_time_stamp = event.timestamp;

   /* If the _hidEvent is null, [event _keyCode] will crash.
    * (This happens with the on screen keyboard). */
   if (event._hidEvent)
   {
      NSString       *ch = (NSString*)event._privateInput;
      uint32_t character = 0;
      uint32_t mod       = 0;
      NSUInteger mods    = event._modifierFlags;

      if (mods & NSAlphaShiftKeyMask)
         mod |= RETROKMOD_CAPSLOCK;
      if (mods & NSShiftKeyMask)
         mod |= RETROKMOD_SHIFT;
      if (mods & NSControlKeyMask)
         mod |= RETROKMOD_CTRL;
      if (mods & NSAlternateKeyMask)
         mod |= RETROKMOD_ALT;
      if (mods & NSCommandKeyMask)
         mod |= RETROKMOD_META;
      if (mods & NSNumericPadKeyMask)
         mod |= RETROKMOD_NUMLOCK;

      if (ch && ch.length != 0)
      {
         unsigned i;
         character = [ch characterAtIndex:0];

         apple_input_keyboard_event(event._isKeyDown,
               (uint32_t)event._keyCode, 0, mod,
               RETRO_DEVICE_KEYBOARD);

         for (i = 1; i < ch.length; i++)
            apple_input_keyboard_event(event._isKeyDown,
                  0, [ch characterAtIndex:i], mod,
                  RETRO_DEVICE_KEYBOARD);
      }

      apple_input_keyboard_event(event._isKeyDown,
            (uint32_t)event._keyCode, character, mod,
            RETRO_DEVICE_KEYBOARD);
   }

   return [super _keyCommandForEvent:event];
}
#else
- (void)handleUIPress:(UIPress *)press withEvent:(UIPressesEvent *)event down:(BOOL)down
{
   NSString       *ch;
   uint32_t character = 0;
   uint32_t mod       = 0;
   NSUInteger mods    = 0;
   if (@available(iOS 13.4, tvOS 13.4, *))
   {
      ch = (NSString*)press.key.characters;
      mods = event.modifierFlags;
   }

   if (mods & UIKeyModifierAlphaShift)
      mod |= RETROKMOD_CAPSLOCK;
   if (mods & UIKeyModifierShift)
      mod |= RETROKMOD_SHIFT;
   if (mods & UIKeyModifierControl)
      mod |= RETROKMOD_CTRL;
   if (mods & UIKeyModifierAlternate)
      mod |= RETROKMOD_ALT;
   if (mods & UIKeyModifierCommand)
      mod |= RETROKMOD_META;
   if (mods & UIKeyModifierNumericPad)
      mod |= RETROKMOD_NUMLOCK;

   if (ch && ch.length != 0)
   {
      unsigned i;
      character = [ch characterAtIndex:0];

      apple_input_keyboard_event(down,
                                 (uint32_t)press.key.keyCode, 0, mod,
                                 RETRO_DEVICE_KEYBOARD);

      for (i = 1; i < ch.length; i++)
         apple_input_keyboard_event(down,
                                    0, [ch characterAtIndex:i], mod,
                                    RETRO_DEVICE_KEYBOARD);
   }

   if (@available(iOS 13.4, tvOS 13.4, *))
      apple_input_keyboard_event(down,
                                 (uint32_t)press.key.keyCode, character, mod,
                                 RETRO_DEVICE_KEYBOARD);
}

- (void)pressesBegan:(NSSet<UIPress *> *)presses withEvent:(UIPressesEvent *)event
{
   for (UIPress *press in presses)
      [self handleUIPress:press withEvent:event down:YES];
   [super pressesBegan:presses withEvent:event];
}

- (void)pressesEnded:(NSSet<UIPress *> *)presses withEvent:(UIPressesEvent *)event
{
   for (UIPress *press in presses)
      [self handleUIPress:press withEvent:event down:NO];
   [super pressesEnded:presses withEvent:event];
}
#endif

#define GSEVENT_TYPE_KEYDOWN 10
#define GSEVENT_TYPE_KEYUP 11

- (void)sendEvent:(UIEvent *)event
{
   [super sendEvent:event];
   if (@available(iOS 13.4, tvOS 13.4, *))
   {
      if (event.type == UIEventTypeHover)
         return;
   }
   if (event.allTouches.count)
      handle_touch_event(event.allTouches.allObjects);

#if __IPHONE_OS_VERSION_MAX_ALLOWED < 70000
   {
      int major, minor;
      get_ios_version(&major, &minor);

      if ((major < 7) && [event respondsToSelector:@selector(_gsEvent)])
      {
         /* Keyboard event hack for iOS versions prior to iOS 7.
          *
          * Derived from:
                  * http://nacho4d-nacho4d.blogspot.com/2012/01/
                  * catching-keyboard-events-in-ios.html
                  */
         const uint8_t *eventMem = objc_unretainedPointer([event performSelector:@selector(_gsEvent)]);
         int           eventType = eventMem ? *(int*)&eventMem[8] : 0;

         switch (eventType)
         {
            case GSEVENT_TYPE_KEYDOWN:
              case GSEVENT_TYPE_KEYUP:
               apple_input_keyboard_event(eventType == GSEVENT_TYPE_KEYDOWN,
                     *(uint16_t*)&eventMem[0x3C], 0, 0, RETRO_DEVICE_KEYBOARD);
               break;
         }
      }
   }
#endif
}

@end

#ifdef HAVE_COCOA_METAL
@implementation MetalLayerView

+ (Class)layerClass {
    return [CAMetalLayer class];
}

- (instancetype)init {
    self = [super init];
    if (self)
        [self setupMetalLayer];
    return self;
}

- (instancetype)initWithFrame:(CGRect)frame {
    self = [super initWithFrame:frame];
    if (self)
        [self setupMetalLayer];
    return self;
}

- (CAMetalLayer *)metalLayer {
    return (CAMetalLayer *)self.layer;
}

- (void)setupMetalLayer {
    self.metalLayer.device = MTLCreateSystemDefaultDevice();
    self.metalLayer.contentsScale = cocoa_screen_get_native_scale();
    self.metalLayer.opaque = YES;
}

@end
#endif

#if TARGET_OS_IOS
@interface RetroArch_iOS () <MXMetricManagerSubscriber, UIPointerInteractionDelegate>
@end
#endif

@interface RetroArch_iOS () <UITextFieldDelegate>
@property (nonatomic, strong) UITextField *keyboardTextField;
@property (nonatomic, copy) void(^keyboardCompletionCallback)(const char *);
@property (nonatomic, assign) char **keyboardBufferPtr;
@property (nonatomic, assign) size_t *keyboardSizePtr;
@property (nonatomic, assign) char *keyboardAllocatedBuffer;
@end

@implementation RetroArch_iOS

#pragma mark - ApplePlatform
-(id)renderView { return _renderView; }
-(bool)hasFocus
{
    return [[UIApplication sharedApplication] applicationState] == UIApplicationStateActive;
}

- (void)setViewType:(apple_view_type_t)vt
{
   if (vt == _vt)
      return;

   _vt = vt;
   if (_renderView != nil)
   {
      [_renderView removeFromSuperview];
      _renderView = nil;
   }

   switch (vt)
   {
#ifdef HAVE_COCOA_METAL
       case APPLE_VIEW_TYPE_VULKAN:
         _renderView = [MetalLayerView new];
#if TARGET_OS_IOS
         _renderView.multipleTouchEnabled = YES;
#endif
         break;
       case APPLE_VIEW_TYPE_METAL:
         {
            MetalView *v = [MetalView new];
            v.paused                = YES;
            v.enableSetNeedsDisplay = NO;
#if TARGET_OS_IOS
            v.multipleTouchEnabled  = YES;
#endif
            _renderView = v;
         }
         break;
#endif
       case APPLE_VIEW_TYPE_OPENGL_ES:
         _renderView = (BRIDGE GLKView*)glkitview_init();
         break;

       case APPLE_VIEW_TYPE_NONE:
       default:
         return;
   }

   _renderView.translatesAutoresizingMaskIntoConstraints = NO;
   UIView *rootView = [CocoaView get].view;
   [rootView addSubview:_renderView];
#if TARGET_OS_IOS
   if (@available(iOS 13.4, *))
   {
      [_renderView addInteraction:[[UIPointerInteraction alloc] initWithDelegate:self]];
      _renderView.userInteractionEnabled = YES;
   }
#endif
   [[_renderView.topAnchor constraintEqualToAnchor:rootView.topAnchor] setActive:YES];
   [[_renderView.bottomAnchor constraintEqualToAnchor:rootView.bottomAnchor] setActive:YES];
   [[_renderView.leadingAnchor constraintEqualToAnchor:rootView.leadingAnchor] setActive:YES];
   [[_renderView.trailingAnchor constraintEqualToAnchor:rootView.trailingAnchor] setActive:YES];
   [_renderView layoutIfNeeded];
}

- (apple_view_type_t)viewType { return _vt; }

- (void)setVideoMode:(gfx_ctx_mode_t)mode
{
#ifdef HAVE_COCOA_METAL
   MetalView *metalView = (MetalView*) _renderView;
   CGFloat scale        = [[UIScreen mainScreen] scale];
   [metalView setDrawableSize:CGSizeMake(
         _renderView.bounds.size.width * scale,
         _renderView.bounds.size.height * scale
         )];
#endif
}

- (void)setCursorVisible:(bool)v { /* no-op for iOS */ }
- (bool)setDisableDisplaySleep:(bool)disable
{
#if TARGET_OS_TV
   [[UIApplication sharedApplication] setIdleTimerDisabled:disable];
   return YES;
#else
   return NO;
#endif
}
+ (RetroArch_iOS*)get { return (RetroArch_iOS*)[[UIApplication sharedApplication] delegate]; }

-(NSString*)documentsDirectory
{
   if (_documentsDirectory == nil)
   {
#if TARGET_OS_IOS
      NSArray *paths      = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
#elif TARGET_OS_TV
      NSArray *paths      = NSSearchPathForDirectoriesInDomains(NSCachesDirectory, NSUserDomainMask, YES);
#endif
      _documentsDirectory = paths.firstObject;
   }
   return _documentsDirectory;
}

- (void)handleAudioSessionInterruption:(NSNotification *)notification
{
   NSNumber *type = notification.userInfo[AVAudioSessionInterruptionTypeKey];
   if (![type isKindOfClass:[NSNumber class]])
      return;

   if ([type unsignedIntegerValue] == AVAudioSessionInterruptionTypeBegan)
   {
      RARCH_DBG("[Cocoa] AudioSession Interruption Began.\n");
      audio_driver_stop();
   }
   else if ([type unsignedIntegerValue] == AVAudioSessionInterruptionTypeEnded)
   {
      RARCH_DBG("[Cocoa] AudioSession Interruption Ended.\n");
      audio_driver_start(false);
   }
}

#ifdef HAVE_KSCRASH
- (NSString *)crashReportsPath
{
   /* Store crash reports in Documents directory for user access */
   NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
   NSString *documentsPath = [paths firstObject];
   return [documentsPath stringByAppendingPathComponent:@"CrashReports"];
}

- (void)initKSCrash
{
   NSString *crashReportsPath = [self crashReportsPath];

   /* Create the crash reports directory if it doesn't exist */
   NSFileManager *fileManager = [NSFileManager defaultManager];
   NSError *createError = nil;
   if (![fileManager fileExistsAtPath:crashReportsPath])
   {
      [fileManager createDirectoryAtPath:crashReportsPath
             withIntermediateDirectories:YES
                              attributes:nil
                                   error:&createError];
      if (createError)
      {
         NSLog(@"[KSCrash] Failed to create crash reports directory: %@\n", createError);
         return;
      }
   }

   /* Configure KSCrash for local storage only */
   KSCrashConfiguration *config = [KSCrashConfiguration new];
   config.installPath = crashReportsPath;
   KSCrashReportStoreConfiguration *storeConfig = [KSCrashReportStoreConfiguration new];
   storeConfig.reportsPath = crashReportsPath;
   storeConfig.appName = @"RetroArch";
   storeConfig.maxReportCount = 10; /* Keep last 10 crash reports */
   config.reportStoreConfiguration = storeConfig;

   /* Set appropriate monitors */
   if (jit_available())
      config.monitors = KSCrashMonitorTypeDebuggerSafe;
   else
      config.monitors = KSCrashMonitorTypeProductionSafe;
   /* Enable useful debugging features */
   config.enableMemoryIntrospection = YES;
   config.enableQueueNameSearch = YES;
   config.addConsoleLogToReport = YES;

   /* Install KSCrash without any network sink */
   NSError *installError = nil;
   if (![[KSCrash sharedInstance] installWithConfiguration:config error:&installError])
   {
      NSLog(@"[KSCrash] Failed to install crash reporter: %@\n", installError);
      return;
   }

   NSLog(@"[KSCrash] reports will be stored in: %@\n", crashReportsPath);
}

- (void)processKSCrashReports
{
   /* Check if we crashed last launch */
   if (![[KSCrash sharedInstance] crashedLastLaunch])
       return;

   if ([[[KSCrash sharedInstance] reportStore] reportCount] <= 0)
      return;

   RARCH_LOG("[KSCrash] crash report available in Documents/CrashReports\n");

   /* Process crash reports to strip binary_images section */
   KSCrashReportStore *store = [[KSCrash sharedInstance] reportStore];
   NSArray<NSNumber *> *reportIDs = [store reportIDs];
   for (NSNumber *reportIDNum in reportIDs)
   {
      int64_t reportID = [reportIDNum longLongValue];
      KSCrashReportDictionary *report = [store reportForID:reportID];
      if (!report)
         continue;

      NSMutableDictionary *mutableReport = [report.value mutableCopy];

      /* Remove binary_images to reduce file size */
      if ([mutableReport objectForKey:@"binary_images"])
         [mutableReport removeObjectForKey:@"binary_images"];

      /* Save pretty-printed version as standalone file */
      NSData *prettyData = [NSJSONSerialization dataWithJSONObject:mutableReport
                                                           options:NSJSONWritingPrettyPrinted
                                                             error:nil];
      if (prettyData)
      {
         NSString *reportPath = [NSString stringWithFormat:@"%@/report-%lld.json",
                                 [self crashReportsPath], reportID];
         [prettyData writeToFile:reportPath options:NSDataWritingAtomic error:nil];
         RARCH_LOG("[KSCrash] Saved stripped report %lld to: %s\n",
                   reportID, [reportPath UTF8String]);
      }

      /* Log minified JSON on a single line for easy extraction */
      NSData *minifiedData = [NSJSONSerialization dataWithJSONObject:mutableReport
                                                             options:0  /* no pretty printing */
                                                               error:nil];
      if (minifiedData)
      {
         NSString *jsonString = [[NSString alloc] initWithData:minifiedData
                                                      encoding:NSUTF8StringEncoding];
         if (jsonString)
         {
            /* Log with a unique marker that can be extracted with grep/sed */
            RARCH_LOG("[KSCrash] Report %lld follows on next line\n", reportID);
            RARCH_LOG("%s\n", [jsonString UTF8String]);
         }
      }

      /* Delete the report from KSCrash store to prevent re-logging on next launch */
      [store deleteReportWithID:reportID];
   }

   if (reportIDs.count > 0)
      RARCH_LOG("[KSCrash] Processed and removed %lu report(s) from store\n", (unsigned long)reportIDs.count);
}
#endif

- (void)applicationDidFinishLaunching:(UIApplication *)application
{
#ifdef HAVE_KSCRASH
   [self initKSCrash];
#endif

   char arguments[]   = "retroarch";
   char       *argv[] = {arguments,   NULL};
   int argc           = 1;
   apple_platform     = self;

   if ([NSUserDefaults.standardUserDefaults boolForKey:@"restore_default_config"])
   {
      [NSUserDefaults.standardUserDefaults setBool:NO forKey:@"restore_default_config"];
      [NSUserDefaults.standardUserDefaults setObject:@"" forKey:@FILE_PATH_MAIN_CONFIG];

      // Get the Caches directory path
      NSArray *paths = NSSearchPathForDirectoriesInDomains(NSCachesDirectory, NSUserDomainMask, YES);
      NSString *cachesDirectory = [paths firstObject];

      // Define the original and new file paths
      NSString *originalPath = [cachesDirectory stringByAppendingPathComponent:@"RetroArch/config/retroarch.cfg"];
      NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
      [dateFormatter setDateFormat:@"HHmm-yyMMdd"];
      NSString *timestamp = [dateFormatter stringFromDate:[NSDate date]];
      NSString *newPath = [cachesDirectory stringByAppendingPathComponent:[NSString stringWithFormat:@"RetroArch/config/RetroArch-%@.cfg", timestamp]];

      // File manager instance
      NSFileManager *fileManager = [NSFileManager defaultManager];

      // Check if the file exists and rename it
      if ([fileManager fileExistsAtPath:originalPath])
      {
          NSError *error = nil;
          if ([fileManager moveItemAtPath:originalPath toPath:newPath error:&error])
              NSLog(@"File renamed to %@", newPath);
          else
              NSLog(@"Error renaming file: %@", error.localizedDescription);
      }
      else
          NSLog(@"File does not exist at path %@", originalPath);
   }

   [self setDelegate:self];

   /* Setup window */
   self.window        = [[UIWindow alloc] initWithFrame:[[UIScreen mainScreen] bounds]];
   [self.window makeKeyAndVisible];

   [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleAudioSessionInterruption:) name:AVAudioSessionInterruptionNotification object:[AVAudioSession sharedInstance]];

   [self showGameView];

   rarch_main(argc, argv, NULL);

#ifdef HAVE_KSCRASH
   [self processKSCrashReports];
#endif

   uico_driver_state_t *uico_st     = uico_state_get_ptr();
   rarch_setting_t *appicon_setting = menu_setting_find_enum(MENU_ENUM_LABEL_APPICON_SETTINGS);
   struct string_list *icons;
   if (               appicon_setting
           && uico_st->drv
           && uico_st->drv->get_app_icons
           && (icons = uico_st->drv->get_app_icons())
           && icons->size > 1)
   {
      int i;
      size_t _len    = 0;
      char *options = NULL;
      const char *icon_name;

      appicon_setting->default_value.string = icons->elems[0].data;
      icon_name = [[application alternateIconName] cStringUsingEncoding:kCFStringEncodingUTF8]; /* need to ask uico_st for this */
      for (i = 0; i < (int)icons->size; i++)
      {
         _len += strlen(icons->elems[i].data) + 1;
         if (string_is_equal(icon_name, icons->elems[i].data))
            appicon_setting->value.target.string = icons->elems[i].data;
      }
      options = (char*)calloc(_len, sizeof(char));
      string_list_join_concat(options, _len, icons, "|");
      if (appicon_setting->values)
         free((void*)appicon_setting->values);
      appicon_setting->values = options;
   }

#if TARGET_OS_TV
   update_topshelf();
#endif

#if HAVE_SWIFT
   if (@available(iOS 16.0, tvOS 16.0, *)) {
      [RetroArchAppShortcuts updateAppShortcuts];
   }
#endif

#if TARGET_OS_IOS
   if (@available(iOS 13.0, *))
      [MXMetricManager.sharedManager addSubscriber:self];
#endif

#ifdef HAVE_MFI
   extern void *apple_gamecontroller_joypad_init(void *data);
   apple_gamecontroller_joypad_init(NULL);
   if (@available(macOS 11, iOS 14, tvOS 14, *))
   {
      [[NSNotificationCenter defaultCenter] addObserverForName:GCMouseDidConnectNotification
                                                        object:nil
                                                         queue:[NSOperationQueue mainQueue]
                                                    usingBlock:^(NSNotification *note)
       {
         GCMouse *mouse = note.object;
         mouse.mouseInput.mouseMovedHandler = ^(GCMouseInput * _Nonnull mouse, float delta_x, float delta_y)
         {
            cocoa_input_data_t *apple = (cocoa_input_data_t*) input_state_get_ptr()->current_data;
            if (!apple)
               return;
            apple->window_pos_x      += (int16_t)delta_x;
            apple->window_pos_y      -= (int16_t)delta_y;
         };
         mouse.mouseInput.leftButton.pressedChangedHandler = ^(GCControllerButtonInput * _Nonnull button, float value, BOOL pressed)
         {
            cocoa_input_data_t *apple = (cocoa_input_data_t*) input_state_get_ptr()->current_data;
            if (!apple)
               return;
            if (pressed)
                apple->mouse_buttons |= (1 << 0);
            else
                apple->mouse_buttons &= ~(1 << 0);
         };
         mouse.mouseInput.rightButton.pressedChangedHandler = ^(GCControllerButtonInput * _Nonnull button, float value, BOOL pressed)
         {
            cocoa_input_data_t *apple = (cocoa_input_data_t*) input_state_get_ptr()->current_data;
            if (!apple)
               return;
            if (pressed)
                apple->mouse_buttons |= (1 << 1);
            else
                apple->mouse_buttons &= ~(1 << 1);
         };
      }];
   }
#endif
}

- (void)applicationDidEnterBackground:(UIApplication *)application
{
#if TARGET_OS_TV
   update_topshelf();
#endif
   rarch_stop_draw_observer();
   command_event(CMD_EVENT_SAVE_FILES, NULL);

   /* Clear any stuck or stale touches when backgrounding */
   cocoa_input_data_t *apple = (cocoa_input_data_t*)input_state_get_ptr()->current_data;
   if (apple)
   {
      apple->touch_count = 0;
      memset(apple->touches, 0, sizeof(apple->touches));
   }
}

- (void)applicationWillTerminate:(UIApplication *)application
{
   rarch_stop_draw_observer();
   retroarch_main_quit();
}

- (void)applicationWillResignActive:(UIApplication *)application
{
   self.bgDate = [NSDate date];
   rarch_stop_draw_observer();

   /* Clear any stuck or stale touches when losing focus */
   cocoa_input_data_t *apple = (cocoa_input_data_t*)input_state_get_ptr()->current_data;
   if (apple)
   {
      apple->touch_count = 0;
      memset(apple->touches, 0, sizeof(apple->touches));
   }
}

- (void)applicationDidBecomeActive:(UIApplication *)application
{
   NSError *error;
   settings_t *settings            = config_get_ptr();
   bool ui_companion_start_on_boot = settings->bools.ui_companion_start_on_boot;

   if (settings->bools.audio_respect_silent_mode)
       [[AVAudioSession sharedInstance] setCategory:AVAudioSessionCategoryAmbient error:&error];
   else
       [[AVAudioSession sharedInstance] setCategory:AVAudioSessionCategoryPlayback error:&error];

   if (!ui_companion_start_on_boot)
      [self showGameView];

#if TARGET_OS_TV
   rarch_start_draw_observer();
#endif

#ifdef HAVE_CLOUDSYNC
   if (self.bgDate)
   {
      if (   [[NSDate date] timeIntervalSinceDate:self.bgDate] > 60.0f
          && (   !(runloop_get_flags() & RUNLOOP_FLAG_CORE_RUNNING)
              || retroarch_ctl(RARCH_CTL_IS_DUMMY_CORE, NULL)))
         task_push_cloud_sync();
      self.bgDate = nil;
   }
#endif
}

-(BOOL)openRetroArchURL:(NSURL *)url
{
   RARCH_LOG("RetroArch URL received: %s\n", [[url absoluteString] UTF8String]);

   // Handle topshelf URLs: retroarch://topshelf?path=...&core_path=...
   if ([url.host isEqualToString:@"topshelf"])
   {
      NSURLComponents *comp = [[NSURLComponents alloc] initWithURL:url resolvingAgainstBaseURL:NO];
      NSString *ns_path, *ns_core_path;
      char path[PATH_MAX_LENGTH];
      char core_path[PATH_MAX_LENGTH];
      content_ctx_info_t content_info = { 0 };
      for (NSURLQueryItem *q in comp.queryItems)
      {
         if ([q.name isEqualToString:@"path"])
            ns_path = q.value;
         else if ([q.name isEqualToString:@"core_path"])
            ns_core_path = q.value;
      }
      if (!ns_path || !ns_core_path)
         return NO;
      fill_pathname_expand_special(path, [ns_path UTF8String], sizeof(path));
      fill_pathname_expand_special(core_path, [ns_core_path UTF8String], sizeof(core_path));
      RARCH_LOG("[Cocoa] TopShelf told us to open \"%s\" with \"%s\".\n", path, core_path);
      return task_push_load_content_with_new_core_from_companion_ui(core_path, path,
                                                                    NULL, NULL, NULL,
                                                                    &content_info, NULL, NULL);
   }

   // Handle simple start URL: retroarch://start
   if ([url.host isEqualToString:@"start"])
   {
      RARCH_LOG("App shortcut: just starting RetroArch\n");
      return YES; // Just bring app to foreground
   }

   // Handle game launch URL: retroarch://game/filename
   if ([url.host isEqualToString:@"game"])
   {
      NSString *filename = [url.path hasPrefix:@"/"] ? [url.path substringFromIndex:1] : url.path;
      if (filename && filename.length > 0)
      {
         RARCH_LOG("App shortcut: launching game '%s'\n", [filename UTF8String]);
         return cocoa_launch_game_by_filename(filename);
      }
   }

   RARCH_LOG("Unknown RetroArch URL format: %s\n", [[url absoluteString] UTF8String]);
   return NO;
}

-(BOOL)application:(UIApplication *)app openURL:(NSURL *)url options:(NSDictionary<UIApplicationOpenURLOptionsKey, id> *)options {
    if ([[url scheme] isEqualToString:@"retroarch"])
        return [self openRetroArchURL:url];

   NSFileManager *manager = [NSFileManager defaultManager];
   NSString     *filename = (NSString*)url.path.lastPathComponent;
   NSError         *error = nil;
   settings_t *settings   = config_get_ptr();
   char fullpath[PATH_MAX_LENGTH] = {0};
   fill_pathname_join_special(fullpath, settings->paths.directory_core_assets, [filename UTF8String], sizeof(fullpath));
   NSString  *destination = [NSString stringWithUTF8String:fullpath];
   /* Copy file to documents directory if it's not already
    * inside Documents directory */
   if ([url startAccessingSecurityScopedResource])
   {
      if (![[url path] containsString: self.documentsDirectory])
         if (![manager fileExistsAtPath:destination])
            [manager copyItemAtPath:[url path] toPath:destination error:&error];
      [url stopAccessingSecurityScopedResource];
   }
   task_push_dbscan(
      settings->paths.directory_playlist,
      settings->paths.path_content_database,
      fullpath,
      false,
      false,
      NULL);
   return true;
}

- (void)navigationController:(UINavigationController *)navigationController willShowViewController:(UIViewController *)viewController animated:(BOOL)animated
{
#if TARGET_OS_IOS
   [self setToolbarHidden:![[viewController toolbarItems] count] animated:YES];
#endif
}

- (void)showGameView
{
   [self popToRootViewControllerAnimated:NO];

#if TARGET_OS_IOS
   [self setToolbarHidden:true animated:NO];
   [[UIApplication sharedApplication] setStatusBarHidden:true withAnimation:UIStatusBarAnimationNone];
   [[UIApplication sharedApplication] setIdleTimerDisabled:true];
#endif

   [self.window setRootViewController:[CocoaView get]];

   /* Initialize hidden keyboard text field for iOS native keyboard support */
   if (!self.keyboardTextField)
   {
      self.keyboardTextField = [[UITextField alloc] initWithFrame:CGRectMake(0, -100, 1, 1)];
      self.keyboardTextField.delegate = self;
      self.keyboardTextField.autocapitalizationType = UITextAutocapitalizationTypeNone;
      self.keyboardTextField.autocorrectionType = UITextAutocorrectionTypeNo;
      self.keyboardTextField.spellCheckingType = UITextSpellCheckingTypeNo;
      self.keyboardTextField.smartQuotesType = UITextSmartQuotesTypeNo;
      self.keyboardTextField.smartDashesType = UITextSmartDashesTypeNo;
      self.keyboardTextField.smartInsertDeleteType = UITextSmartInsertDeleteTypeNo;
      self.keyboardTextField.returnKeyType = UIReturnKeyDone;
      [[CocoaView get].view addSubview:self.keyboardTextField];
   }

   dispatch_after(dispatch_time(DISPATCH_TIME_NOW, 1.0 * NSEC_PER_SEC), dispatch_get_main_queue(), ^{
         command_event(CMD_EVENT_AUDIO_START, NULL);
         });
}

- (void)supportOtherAudioSessions { }

#if TARGET_OS_IOS
- (void)didReceiveMetricPayloads:(NSArray<MXMetricPayload *> *)payloads API_AVAILABLE(ios(13.0))
{
    for (MXMetricPayload *payload in payloads)
    {
        NSString *json = [[NSString alloc] initWithData:[payload JSONRepresentation] encoding:kCFStringEncodingUTF8];
        RARCH_LOG("[Cocoa] Got Metric Payload:\n%s\n", [json cStringUsingEncoding:kCFStringEncodingUTF8]);
    }
}

- (void)didReceiveDiagnosticPayloads:(NSArray<MXDiagnosticPayload *> *)payloads API_AVAILABLE(ios(14.0))
{
    for (MXDiagnosticPayload *payload in payloads)
    {
        NSString *json = [[NSString alloc] initWithData:[payload JSONRepresentation] encoding:kCFStringEncodingUTF8];
        RARCH_LOG("[Cocoa] Got Diagnostic Payload:\n%s\n", [json cStringUsingEncoding:kCFStringEncodingUTF8]);
    }
}

- (UIPointerStyle *)pointerInteraction:(UIPointerInteraction *)interaction styleForRegion:(UIPointerRegion *)region API_AVAILABLE(ios(13.4))
{
   cocoa_input_data_t *apple = (cocoa_input_data_t*) input_state_get_ptr()->current_data;
   if (!apple)
      return nil;
   if (apple->mouse_grabbed)
      return [UIPointerStyle hiddenPointerStyle];
   return nil;
}

- (UIPointerRegion *)pointerInteraction:(UIPointerInteraction *)interaction
                       regionForRequest:(UIPointerRegionRequest *)request
                          defaultRegion:(UIPointerRegion *)defaultRegion API_AVAILABLE(ios(13.4))
{
   cocoa_input_data_t *apple = (cocoa_input_data_t*) input_state_get_ptr()->current_data;
   if (!apple || apple->mouse_grabbed)
      return nil;
   CGPoint location = [apple_platform.renderView convertPoint:[request location] fromView:nil];
   apple->touches[0].screen_x = (int16_t)(location.x * [[UIScreen mainScreen] scale]);
   apple->touches[0].screen_y = (int16_t)(location.y * [[UIScreen mainScreen] scale]);
   apple->window_pos_x = (int16_t)(location.x * [[UIScreen mainScreen] scale]);
   apple->window_pos_y = (int16_t)(location.y * [[UIScreen mainScreen] scale]);
   return [UIPointerRegion regionWithRect:[apple_platform.renderView bounds] identifier:@"game view"];
}
#endif

#pragma mark - UITextFieldDelegate (iOS/tvOS Native Keyboard Support)

- (BOOL)textField:(UITextField *)textField shouldChangeCharactersInRange:(NSRange)range replacementString:(NSString *)string
{
   if (textField != self.keyboardTextField || !self.keyboardAllocatedBuffer || !self.keyboardSizePtr)
      return YES;

   /* Calculate new text */
   NSString *newText = [textField.text stringByReplacingCharactersInRange:range withString:string];

   /* Update the RetroArch buffer in real-time so the menu can display it */
   const char *utf8Text = [newText UTF8String];
   if (utf8Text)
   {
      strlcpy(self.keyboardAllocatedBuffer, utf8Text, 512);
      *self.keyboardSizePtr = strlen(self.keyboardAllocatedBuffer);
   }

   return YES;
}

- (BOOL)textFieldShouldReturn:(UITextField *)textField
{
   if (textField == self.keyboardTextField)
   {
      /* Update buffer with final text before calling callback */
      if (self.keyboardAllocatedBuffer && self.keyboardSizePtr)
      {
         const char *finalText = [textField.text UTF8String];
         if (finalText)
         {
            strlcpy(self.keyboardAllocatedBuffer, finalText, 512);
            *self.keyboardSizePtr = strlen(self.keyboardAllocatedBuffer);
         }
      }

      /* Store callback and buffer before clearing callback reference */
      void(^callback)(const char *) = self.keyboardCompletionCallback;
      char *buffer = self.keyboardAllocatedBuffer;

      /* Clear callback to prevent double-invoke, but keep buffer references
       * since the callback will free the buffer via input_keyboard_line_free() */
      self.keyboardCompletionCallback = nil;

      /* DON'T dismiss keyboard here - let menu_input_dialog_end() -> ios_keyboard_end() do it
       * This ensures ios_keyboard_active() returns true when the callback checks it */

      /* Call completion callback with buffer pointer
       * The callback will call menu_input_dialog_end() which will call ios_keyboard_end() */
      if (callback && buffer)
         callback(buffer);
      /* Clear our references after callback completes */
      self.keyboardBufferPtr = NULL;
      self.keyboardSizePtr = NULL;
      self.keyboardAllocatedBuffer = NULL;

      return NO;  /* Return NO to prevent UIKit from processing the return key event further */
   }
   return YES;
}

- (void)textFieldDidEndEditing:(UITextField *)textField
{
   if (textField == self.keyboardTextField)
   {
      /* Only call callback if it wasn't already called (by textFieldShouldReturn) */
      if (self.keyboardCompletionCallback)
      {
         /* User dismissed keyboard without hitting return - treat as cancel */
         void(^callback)(const char *) = self.keyboardCompletionCallback;

         /* Clear callback to prevent double-invoke, but keep buffer references
          * since the callback will free the buffer via input_keyboard_line_free() */
         self.keyboardCompletionCallback = nil;

         /* Call callback with NULL to indicate cancel
          * The callback will handle cleanup via input_keyboard_line_free() */
         callback(NULL);

         /* Clear our references after callback completes */
         self.keyboardBufferPtr = NULL;
         self.keyboardSizePtr = NULL;
         self.keyboardAllocatedBuffer = NULL;
      }
   }
}

@end

ui_companion_driver_t ui_companion_cocoatouch = {
   NULL, /* init */
   NULL, /* deinit */
   NULL, /* toggle */
   ui_companion_cocoatouch_event_command,
   NULL, /* notify_refresh */
   NULL, /* msg_queue_push */
   NULL, /* render_messagebox */
   NULL, /* get_main_window */
   NULL, /* log_msg */
   NULL, /* is_active */
   ui_companion_cocoatouch_get_app_icons,
   ui_companion_cocoatouch_set_app_icon,
   ui_companion_cocoatouch_get_app_icon_texture,
   NULL, /* browser_window */
   NULL, /* msg_window */
   NULL, /* window */
   NULL, /* application */
   "cocoatouch",
};

/* C interface for iOS/tvOS native keyboard support */
bool ios_keyboard_start(char **buffer_ptr, size_t *size_ptr, const char *label,
                       input_keyboard_line_complete_t callback, void *userdata)
{
   RetroArch_iOS *app = [RetroArch_iOS get];
   if (!app || !app.keyboardTextField || !buffer_ptr || !size_ptr)
      return false;

   /* Allocate a fixed-size buffer for keyboard input */
   char *allocated_buffer = (char *)malloc(512);
   if (!allocated_buffer)
      return false;

   /* Initialize buffer with existing content if any */
   if (*buffer_ptr && **buffer_ptr)
      strlcpy(allocated_buffer, *buffer_ptr, 512);
   else
      allocated_buffer[0] = '\0';

   /* Update the keyboard_line buffer pointer to point to our allocated buffer */
   *buffer_ptr = allocated_buffer;
   *size_ptr = strlen(allocated_buffer);

   /* Store pointers so we can update them as user types */
   app.keyboardBufferPtr = buffer_ptr;
   app.keyboardSizePtr = size_ptr;
   app.keyboardAllocatedBuffer = allocated_buffer;

   /* Set up the text field with initial text from the buffer */
   app.keyboardTextField.text = (allocated_buffer[0] != '\0') ?
      [NSString stringWithUTF8String:allocated_buffer] : @"";

   /* Optionally set placeholder from label */
   if (label)
      app.keyboardTextField.placeholder = [NSString stringWithUTF8String:label];

   /* Store the completion callback */
   app.keyboardCompletionCallback = ^(const char *text) {
      input_driver_state_t *input_st = input_state_get_ptr();

      if (callback)
         callback(userdata, text);

      /* Clean up RetroArch's keyboard state, mirroring what the built-in keyboard does */
      if (input_st)
      {
         RARCH_LOG("[iOS KB] cleaning up input state\n");
         input_keyboard_line_free(input_st);
         input_st->flags &= ~INP_FLAG_KB_MAPPING_BLOCKED;
      }
   };

   /* Show the keyboard */
   [app.keyboardTextField becomeFirstResponder];
   return true;
}

bool ios_keyboard_active(void)
{
   RetroArch_iOS *app = [RetroArch_iOS get];
   return app && app.keyboardTextField && [app.keyboardTextField isFirstResponder];
}

void ios_keyboard_end(void)
{
   RetroArch_iOS *app = [RetroArch_iOS get];
   if (app && app.keyboardTextField)
   {
      [app.keyboardTextField resignFirstResponder];
      app.keyboardCompletionCallback = nil;
   }
}

int main(int argc, char *argv[])
{
#if TARGET_OS_IOS
    if (jb_enable_ptrace_hack())
        RARCH_LOG("[Cocoa] Ptrace hack complete, JIT support is enabled.\n");
    else
        RARCH_WARN("[Cocoa] Ptrace hack NOT available; Please use an app like Jitterbug.\n");
#endif
#ifdef HAVE_SDL2
    SDL_SetMainReady();
#endif
   @autoreleasepool {
      return UIApplicationMain(argc, argv, NSStringFromClass([RApplication class]), NSStringFromClass([RetroArch_iOS class]));
   }
}
