# -*- coding: utf-8 -*-
#
# Copyright (C) 2012 Steffen Hoffmann <hoff.st@web.de>
# All rights reserved.
#
# This software is licensed as described in the file COPYING, which
# you should have received as part of this distribution.
#
# Author: Steffen Hoffmann <hoff.st@web.de>

import shutil
import string
import tempfile
import unittest

from trac.perm import PermissionCache, PermissionSystem
from trac.util.html import Markup
from trac.test import EnvironmentStub, MockRequest
from trac.web.api import RequestDone
from trac.web.session import Session

from ..api import AccountManager
from ..compat import unicode
from ..db import SessionStore
from ..model import set_user_attribute
from ..register import (
    BasicCheck, BotTrapCheck, EmailCheck, EmailVerificationModule, RegExpCheck,
    RegistrationError, RegistrationModule, UsernamePermCheck)
from . import makeSuite


class _BaseTestCase(unittest.TestCase):
    def setUp(self, method='GET'):
        self.env = EnvironmentStub(
            enable=['trac.*', 'acct_mgr.api.*'])
        self.env.path = tempfile.mkdtemp()
        self.perm = PermissionSystem(self.env)

        # Create a user reference in the permission system.
        self.perm.grant_permission('admin', 'ACCTMGR_USER_ADMIN')
        # Prepare a generic registration request.
        args = dict(username='', name='', email='')
        self.req = MockRequest(self.env,  method=method, path_info='/register',
                               args=args)

    def tearDown(self):
        shutil.rmtree(self.env.path)


class BasicCheckTestCase(_BaseTestCase):
    def setUp(self):
        _BaseTestCase.setUp(self)
        self.env = EnvironmentStub(
            enable=['trac.*', 'acct_mgr.admin.*',
                    'acct_mgr.db.sessionstore',
                    'acct_mgr.pwhash.HtDigestHashMethod'])
        self.env.path = tempfile.mkdtemp()
        self.env.config.set('account-manager', 'password_store',
                            'SessionStore')
        store = SessionStore(self.env)
        store.set_password('registered_user', 'password')

    def test_check(self):
        check = BasicCheck(self.env)
        req = self.req
        # Inspector doesn't provide additional fields.
        field_res = check.render_registration_fields(req, {})
        self.assertEqual(len(field_res), 2)
        self.assertEqual((None, None), field_res)
        # 1st attempt: No input.
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 2nd attempt: Illegal character.
        req.args['username'] = 'user:name'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 3rd attempt: All upper-case word.
        req.args['username'] = 'UPPER-CASE_USER'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 4th attempt: Reserved word.
        req.args['username'] = 'Anonymous'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 5th attempt: Existing user.
        req.args['username'] = 'registered_user'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 6th attempt: Valid username, but no password.
        req.args['username'] = 'user'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 7th attempt: Valid username, no matching passwords.
        req.args['password'] = 'password'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 8th attempt: Finally some valid input.
        req.args['password_confirm'] = 'password'
        self.assertEqual(check.validate_registration(req), None)


class BotTrapCheckTestCase(_BaseTestCase):
    def setUp(self):
        _BaseTestCase.setUp(self)
        self.basic_token = "Yes, I'm human"
        self.env.config.set('account-manager', 'register_basic_token',
                            self.basic_token)

    def test_check(self):
        check = BotTrapCheck(self.env)
        req = self.req

        # Inspector provides the email text input field.
        wrong_input = "Hey, I'm a bot."
        data = dict(basic_token=wrong_input)
        req.args.update(data)
        field_res = check.render_registration_fields(req, data)
        self.assertEqual(len(field_res), 2)
        self.assertTrue(Markup(field_res[0]).startswith('<label>Parole:'))

        # 1st attempt: Wrong input.
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # Ensure, that old input is restored on failure.
        self.assertIn(wrong_input, Markup(field_res[0]))
        # Ensure, that template data dict is passed unchanged.
        self.assertEqual(field_res[1], data)

        # 2nd attempt: No input.
        req.args['basic_token'] = ''
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 3rd attempt: As before, but request done by authenticated user.
        req = MockRequest(self.env, authname='admin', args=self.req.args)
        self.assertEqual(check.validate_registration(req), None)
        # 4th attempt: Finally valid input.
        req = self.req
        req.args['basic_token'] = self.basic_token
        self.assertEqual(check.validate_registration(req), None)
        # 5th attempt: Fill the hidden field too.
        req.args['sentinel'] = "Humans can't see this? Crap - I'm superior!"
        self.assertRaises(RegistrationError, check.validate_registration, req)


class EmailCheckTestCase(_BaseTestCase):
    """Needs several test methods, because dis-/enabling a component doesn't
    work within one method.
    """

    def test_verify_mod_disabled(self):
        """Registration challenges with EmailVerificationModule disabled."""
        self.env = EnvironmentStub(
            enable=['trac.*', 'acct_mgr.admin.*'])
        self.env.path = tempfile.mkdtemp()

        check = EmailCheck(self.env)
        req = self.req

        self.env.config.set('account-manager', 'verify_email', False)
        self.assertEqual(check.validate_registration(req), None)
        # Check should be skipped regardless of AccountManager settings.
        self.env.config.set('account-manager', 'verify_email', True)
        self.assertEqual(check.validate_registration(req), None)

    def test_verify_conf_changes(self):
        """Registration challenges with EmailVerificationModule enabled."""
        self.env = EnvironmentStub(
            enable=['trac.*', 'acct_mgr.admin.*', 'acct_mgr.register.*',
                    'acct_mgr.pwhash.HtDigestHashMethod'])
        self.env.path = tempfile.mkdtemp()
        set_user_attribute(self.env, 'admin', 'email', 'admin@foo.bar')

        check = EmailCheck(self.env)
        req = self.req

        # Inspector provides the email text input field.
        old_email_input = 'email@foo.bar'
        acct = dict(username='user', email=old_email_input, name='User')
        req.args.update(acct)
        field_res = check.render_registration_fields(req, acct)
        self.assertEqual(len(field_res), 2)
        self.assertTrue(Markup(field_res[0]).startswith('<label>Email:'))
        # Ensure, that old input is restored on failure.
        self.assertIn(old_email_input, Markup(field_res[0]))
        # Ensure, that template data dict is passed unchanged.
        self.assertEqual(field_res[1], acct)
        req.args.update(dict(email=''))

        # 1st: Initially try with account verification disabled by setting.
        self.env.config.set('account-manager', 'verify_email', False)
        self.assertEqual(check.validate_registration(req), None)
        # 2nd: Again no email, but now with account verification enabled.
        self.env.config.set('account-manager', 'verify_email', True)
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 3th attempt: Valid email, but already registered with a username.
        req.args['email'] = 'admin@foo.bar'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 4th attempt: Finally some valid input.
        req.args['email'] = 'email@foo.bar'
        self.assertEqual(check.validate_registration(req), None)


class RegExpCheckTestCase(_BaseTestCase):
    def test_check(self):
        self.env = EnvironmentStub(
            enable=['trac.*', 'acct_mgr.admin.*', 'acct_mgr.register.*'])
        self.env.path = tempfile.mkdtemp()

        check = RegExpCheck(self.env)
        req = self.req
        # Inspector doesn't provide additional fields.
        field_res = check.render_registration_fields(req, {})
        self.assertEqual(len(field_res), 2)
        self.assertEqual((None, None), field_res)
        # Empty input is invalid with default regular expressions.
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 1st attempt: Malformed email address.
        req.args['username'] = 'username'
        req.args['email'] = 'email'
        self.assertRaises(RegistrationError, check.validate_registration,
                          req)
        # 2nd attempt: Same as before, but with email verification disabled.
        self.env.config.set('account-manager', 'verify_email', False)
        self.assertEqual(check.validate_registration(req), None)
        # 3rd attempt: Now with email verification enabled, but valid email.
        self.env.config.set('account-manager', 'verify_email', True)
        req.args['email'] = 'email@foo.bar'
        self.assertEqual(check.validate_registration(req), None)
        # 4th attempt: Now with too short username.
        req.args['username'] = 'user'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 5th attempt: Allow short username by custom regular expression.
        self.env.config.set('account-manager', 'username_regexp',
                            r'(?i)^[A-Z.\-_]{4,}$')
        self.assertEqual(check.validate_registration(req), None)
        # 6th attempt: Now with username containing single quote.
        req.args['username'] = 'user\'name'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 7th attempt: Alter config to allow username containing single quote.
        self.env.config.set('account-manager', 'username_regexp',
                            r'(?i)^[A-Z.\-\'_]{4,}$')
        self.assertEqual(check.validate_registration(req), None)


class UsernamePermCheckTestCase(_BaseTestCase):
    def test_check(self):
        check = UsernamePermCheck(self.env)
        req = self.req
        # Inspector doesn't provide additional fields.
        field_res = check.render_registration_fields(req, {})
        self.assertEqual(len(field_res), 2)
        self.assertEqual((None, None), field_res)
        # 1st attempt: No username, hence no conflic possible.
        self.assertEqual(check.validate_registration(req), None)
        # 2nd attempt: No permission assigned for that username.
        req.args['username'] = 'user'
        self.assertEqual(check.validate_registration(req), None)
        # 3rd attempt: Explicit permission assigned for that username.
        req.args['username'] = 'admin'
        self.assertRaises(RegistrationError, check.validate_registration, req)
        # 4th attempt: As before, but request done by authenticated user.
        req = MockRequest(self.env, authname='admin', args=self.req.args)
        self.assertEqual(check.validate_registration(req), None)


class RegistrationModuleTestCase(_BaseTestCase):
    def setUp(self):
        _BaseTestCase.setUp(self, method='POST')
        self.env = EnvironmentStub(enable=[
            'trac.*', 'acct_mgr.api.*', 'acct_mgr.db.*',
            'acct_mgr.register.*',
            'acct_mgr.pwhash.HtDigestHashMethod'
        ])
        self.env.path = tempfile.mkdtemp()
        self.reg_template = 'account_register.html'

        self.env.config.set('account-manager', 'password_store',
                            'SessionStore')
        self.acctmgr = AccountManager(self.env)
        self.check = BasicCheck(self.env)
        self.rmod = RegistrationModule(self.env)
        self.store = SessionStore(self.env)

    def test_checks(self):
        # Default configuration: All default checks enabled.
        response = self.rmod.process_request(self.req)
        self.assertEqual(response[0], self.reg_template)

        # Custom configuration: Do basic username checks only.
        self.req.args['username'] = 'admin'
        self.req.args['email'] = 'admin@foo.bar'
        self.env.config.set('account-manager', 'register_check',
                            'BasicCheck')
        response = self.rmod.process_request(self.req)
        self.assertEqual(response[0], self.reg_template)

        # Custom configuration: No check at all, if you insist.
        self.env.config.set('account-manager', 'register_check', '')
        self.assertFalse(self.acctmgr.register_checks)
        response = self.rmod.process_request(self.req)
        self.assertEqual(response[0], self.reg_template)

    def test_mandatory_email_registration(self):
        user = 'user1'
        passwd = 'test'
        # A more complete mock of a request object is required for this test.
        req = MockRequest(self.env, method='POST', args={
            'action': 'create',
            'username': user,
            'name': 'Tester',
            'password': passwd,
            'password_confirm': passwd
        })
        # Fail to register the user.
        self.rmod.process_request(req)
        self.assertIn('email address', unicode(req.chrome['warnings']))
        self.assertEqual(list(self.store.get_users()), [])

    def test_optional_email_registration(self):
        user = 'user1'
        passwd = 'test'

        req = MockRequest(self.env, path_info='/register', method='POST',
                args={
                    'action': 'create',
                    'username': user,
                    'name': 'Tester',
                    'password': passwd,
                    'password_confirm': passwd
        })
        self.env.config.set('account-manager', 'verify_email', False)
        # Successfully register the user.
        # Note: This would have raised an AttributeError without graceful
        #   request checking for 'email'.
        self.assertRaises(RequestDone, self.rmod.process_request, req)
        # DEVEL: Check registration success more directly.
        self.assertEqual(req.chrome['warnings'], [])
        self.assertEqual([user], list(self.store.get_users()))
        self.assertTrue(self.store.check_password(user, passwd))


class EmailVerificationModuleTestCase(_BaseTestCase):
    """Verify email address validation when running account verification."""

    def setUp(self):
        _BaseTestCase.setUp(self)
        self.env = EnvironmentStub(
            enable=['trac.*', 'acct_mgr.api.*', 'acct_mgr.register.*'])
        self.env.path = tempfile.mkdtemp()

        args = dict(username='user', name='', email='')
        self.req = MockRequest(self.env, authname='user', path_info='/prefs',
                               method='POST', args=args)
        self.req.session = Session(self.env, self.req)
        self.req.session['email'] = 'email@foo.bar'
        self.req.session.save()
        self.vmod = EmailVerificationModule(self.env)

    def test_check_email_used(self):
        set_user_attribute(self.env, 'admin', 'email', 'admin@foo.bar')
        # Try email, that is already associated to another user.
        self.req.args['email'] = 'admin@foo.bar'
        self.assertRaises(RequestDone, self.vmod.pre_process_request,
                          self.req, None)
        warnings = self.req.chrome.get('warnings')
        self.assertIn('already in use',
                      unicode(warnings and warnings[0] or ''))

    def test_check_no_email(self):
        self.assertRaises(RequestDone, self.vmod.pre_process_request,
                          self.req, None)
        warnings = self.req.chrome.get('warnings')
        self.assertNotEqual(unicode(warnings and warnings[0] or ''), '')

    def test_check(self):
        self.req.args['email'] = 'user@foo.bar'
        self.vmod.pre_process_request(self.req, None)
        warnings = self.req.chrome.get('warnings')
        self.assertEqual(unicode(warnings and warnings[0] or ''), '')


def test_suite():
    suite = unittest.TestSuite()
    suite.addTest(makeSuite(BasicCheckTestCase))
    suite.addTest(makeSuite(BotTrapCheckTestCase))
    suite.addTest(makeSuite(EmailCheckTestCase))
    suite.addTest(makeSuite(RegExpCheckTestCase))
    suite.addTest(makeSuite(UsernamePermCheckTestCase))
    suite.addTest(makeSuite(RegistrationModuleTestCase))
    suite.addTest(makeSuite(EmailVerificationModuleTestCase))
    return suite


if __name__ == '__main__':
    unittest.main(defaultTest='test_suite')
