/**
 * \file pappsomspp/processing/cbor/cborstreamwriter.h
 * \date 05/07/2025
 * \author Olivier Langella
 * \brief PAPPSO CBOR stream writer
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#pragma once

#include <QCborStreamWriter>
#include <QCborMap>


#include "pappsomspp/core/msrun/xiccoord/xiccoord.h"
#include "pappsomspp/export-import-config.h"
#include "pappsomspp/core/trace/trace.h"


namespace pappso
{
namespace cbor
{
/** @brief overrides QCborStreamWriter base class to provide convenient functions
 */
class PMSPP_LIB_DECL CborStreamWriter : public QCborStreamWriter
{
  public:
  /**
   * Default constructor
   */
  CborStreamWriter(QByteArray *data);
  CborStreamWriter(QIODevice *device);

  /**
   * Destructor
   */
  virtual ~CborStreamWriter();


  /** @brief get UUID intialized when using @ref writeInformations
   * @return QString for the UUID
   */
  const QString &getUuid() const;

  /** @brief automatically produces an informations CBOR map
   * the "informations" map contains default parameters : software, versions, UUID
   */
  void writeInformations(const QString &software_name,
                         const QString &software_version,
                         const QString &type,
                         const QString &operation);


  void writeCborMap(const QCborMap &cbor_map);
  void writeCborArray(const QCborArray &cbor_array);
  void writeArray(const std::vector<std::size_t> &int_list);
  void writeArray(const std::vector<qint64> &int_list);
  void writeArray(const std::vector<double> &double_list);
  void writeArray(const std::vector<int> &positions);
  void writeArray(const std::vector<std::uint8_t> &small_uint_list);
  void writeArray(const QStringList &str_list);
  void writeArray(const std::vector<QString> &str_list);

  /** @brief serialize a PAPPSO trace object to CBOR
   * @param trace PAPPSO trace object reference
   * @param x_label label for first dimension (x by default)
   * @param y_label label for second dimension (y by default)
   */
  void writeTrace(const pappso::Trace &trace,
                  const QString &x_label = "x",
                  const QString &y_label = "y");


  /** @brief serialize a PAPPSO xic coordinate object to CBOR
   * @param xic PAPPSO object reference
   */
  void writeXicCoord(const pappso::XicCoord &xic_coord);

  /** @brief serialize a PAPPSO xic coordinate object list to CBOR
   * @param xic_coord_list PAPPSO object reference
   */
  void writeXicCoordList(const std::vector<pappso::XicCoordSPtr> &xic_coord_list);

  private:
  QString m_uuid;
};
} // namespace cbor
} // namespace pappso
